<?php
/**
 * AIOX Content Data Class
 * 
 * Handles content data extraction with ingredients system
 */

if (!defined('ABSPATH')) {
    exit;
}

class AIOX_Content_Data {
    
    /**
     * Get all content data with ingredients
     */
    public static function get_all_content_data_with_ingredients() {
        // First, fix any existing processed posts that don't have the _aiox_processed flag
        self::fix_existing_processed_posts();
        
        // Get posts that have been processed (either with ingredients or Q&A)
        $posts = get_posts(array(
            'post_type' => array('post', 'page'),
            'post_status' => 'publish',
            'numberposts' => -1,
            'meta_query' => array(
                'relation' => 'OR',
                array(
                    'key' => '_aiox_processed',
                    'value' => '1',
                    'compare' => '='
                ),
                array(
                    'key' => '_aiox_processed_at',
                    'compare' => 'EXISTS'
                ),
                array(
                    'key' => '_aiox_ingredients',
                    'compare' => 'EXISTS'
                )
            )
        ));
        
        $content_data = array();
        
        foreach ($posts as $post) {
            $post_data = self::get_post_content_data_with_ingredients($post);
            if ($post_data) {
                $content_data[] = $post_data;
            }
        }
        
        return $content_data;
    }
    
    /**
     * Fix existing processed posts that don't have the _aiox_processed flag
     */
    public static function fix_existing_processed_posts() {
        global $wpdb;
        
        // Find posts that have ingredients or processed_at but no _aiox_processed flag
        $posts_to_fix = $wpdb->get_results("
            SELECT DISTINCT p.ID 
            FROM {$wpdb->posts} p
            INNER JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id
            WHERE p.post_status = 'publish' 
            AND p.post_type IN ('post', 'page')
            AND (pm.meta_key = '_aiox_ingredients' OR pm.meta_key = '_aiox_processed_at')
            AND p.ID NOT IN (
                SELECT post_id FROM {$wpdb->postmeta} 
                WHERE meta_key = '_aiox_processed' AND meta_value = '1'
            )
        ");
        
        foreach ($posts_to_fix as $post) {
            update_post_meta($post->ID, '_aiox_processed', '1');
        }
        
        if (count($posts_to_fix) > 0) {
            AIOX_Logger::info('Fixed existing processed posts', array(
                'count' => count($posts_to_fix),
                'post_ids' => array_column($posts_to_fix, 'ID')
            ));
        }
    }
    
    /**
     * Get content data for a single post with ingredients
     */
    public static function get_post_content_data_with_ingredients($post) {
        if (!$post) {
            return null;
        }
        
        // Get basic post data
        $post_data = array(
            'id' => $post->ID,
            'title' => $post->post_title,
            'url' => get_permalink($post->ID),
            'type' => $post->post_type,
            'status' => $post->post_status,
            'published_at' => $post->post_date,
            'modified_at' => $post->post_modified,
            'author' => get_the_author_meta('display_name', $post->post_author),
            'excerpt' => get_the_excerpt($post->ID),
            'word_count' => str_word_count(wp_strip_all_tags($post->post_content)),
            'reading_time' => ceil(str_word_count(wp_strip_all_tags($post->post_content)) / 200)
        );
        
        // Get enhanced metadata
        $metadata = AIOX_Metadata::get_post_metadata($post->ID);
        $licensing = AIOX_Metadata::get_licensing_metadata($post->ID);
        
        $post_data['metadata'] = $metadata;
        $post_data['licensing'] = $licensing;
        
        // Get ingredients
        $ingredients = AIOX_Ingredients::get_post_ingredients($post->ID);
        $ingredients_summary = AIOX_Ingredients::get_post_ingredients_summary($post->ID);
        
        $post_data['ingredients'] = array(
            'total_count' => $ingredients_summary['total_count'] ?? 0,
            'types' => $ingredients_summary['types'] ?? array(),
            'total_words' => $ingredients_summary['total_words'] ?? 0,
            'last_updated' => $ingredients_summary['last_updated'] ?? null,
            'items' => self::format_ingredients_for_export($ingredients)
        );
        
        // Get legacy Q&A data for compatibility
        $qa_data = get_post_meta($post->ID, '_aiox_qa_data', true);
        if (!empty($qa_data)) {
            $post_data['qa_pairs'] = json_decode($qa_data, true) ?: array();
        } else {
            $post_data['qa_pairs'] = array();
        }
        
        // Add categories and tags
        $post_data['categories'] = wp_get_post_categories($post->ID, array('fields' => 'names'));
        $post_data['tags'] = wp_get_post_tags($post->ID, array('fields' => 'names'));
        
        // Add featured image
        $featured_image = get_the_post_thumbnail_url($post->ID, 'full');
        if ($featured_image) {
            $post_data['featured_image'] = $featured_image;
        }
        
        // Add processing metadata
        $post_data['processing'] = array(
            'processed_at' => get_post_meta($post->ID, '_aiox_processed_at', true),
            'processing_version' => get_post_meta($post->ID, '_aiox_processing_version', true) ?: AIOX_VERSION,
            'ingredients_enabled' => get_option('aiox_enable_ingredients_system', true),
            'metadata_classification_enabled' => get_option('aiox_enable_metadata_classification', true)
        );
        
        return $post_data;
    }
    
    /**
     * Format ingredients for export
     */
    private static function format_ingredients_for_export($ingredients) {
        if (empty($ingredients)) {
            return array();
        }
        
        $formatted_ingredients = array();
        
        foreach ($ingredients as $ingredient) {
            $formatted_ingredient = array(
                'id' => $ingredient['id'],
                'type' => $ingredient['type'],
                'content' => $ingredient['content'],
                'word_count' => $ingredient['word_count'],
                'character_count' => $ingredient['character_count'],
                'language' => $ingredient['language'],
                'created_at' => $ingredient['created_at']
            );
            
            // Add context if available
            if (!empty($ingredient['context'])) {
                $formatted_ingredient['context'] = $ingredient['context'];
            }
            
            // Add metadata summary (not full metadata to avoid duplication)
            if (!empty($ingredient['metadata'])) {
                $formatted_ingredient['content_metadata'] = array(
                    'intent' => $ingredient['metadata']['intent'] ?? 'informational',
                    'tone' => $ingredient['metadata']['tone'] ?? 'professional',
                    'target_audience' => $ingredient['metadata']['target_audience'] ?? 'general',
                    'utility' => $ingredient['metadata']['utility'] ?? 'reference'
                );
            }
            
            // Add licensing summary
            if (!empty($ingredient['licensing'])) {
                $formatted_ingredient['licensing'] = array(
                    'training_allowed' => $ingredient['licensing']['training_allowed'] ?? false,
                    'citation_required' => $ingredient['licensing']['citation_required'] ?? true,
                    'attribution_string' => $ingredient['licensing']['attribution_string'] ?? get_bloginfo('name')
                );
            }
            
            $formatted_ingredients[] = $formatted_ingredient;
        }
        
        return $formatted_ingredients;
    }
    
    /**
     * Get total processed posts count
     */
    public static function get_total_processed_posts() {
        global $wpdb;
        
        $count = $wpdb->get_var("
            SELECT COUNT(*) 
            FROM {$wpdb->posts} p
            INNER JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id
            WHERE p.post_status = 'publish'
            AND p.post_type IN ('post', 'page')
            AND pm.meta_key = '_aiox_processed'
            AND pm.meta_value = '1'
        ");
        
        return intval($count);
    }
    
    /**
     * Get total Q&A pairs count (legacy compatibility)
     */
    public static function get_total_qa_pairs() {
        global $wpdb;
        
        $qa_data = $wpdb->get_results("
            SELECT pm.meta_value
            FROM {$wpdb->posts} p
            INNER JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id
            WHERE p.post_status = 'publish'
            AND p.post_type IN ('post', 'page')
            AND pm.meta_key = '_aiox_qa_data'
            AND pm.meta_value != ''
        ");
        
        $total_qa_pairs = 0;
        
        foreach ($qa_data as $data) {
            $qa_array = json_decode($data->meta_value, true);
            if (is_array($qa_array)) {
                $total_qa_pairs += count($qa_array);
            }
        }
        
        return $total_qa_pairs;
    }
    
    /**
     * Get total ingredients count
     */
    public static function get_total_ingredients() {
        global $wpdb;
        
        $count = $wpdb->get_var("
            SELECT SUM(CAST(pm.meta_value AS UNSIGNED))
            FROM {$wpdb->posts} p
            INNER JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id
            WHERE p.post_status = 'publish'
            AND p.post_type IN ('post', 'page')
            AND pm.meta_key = '_aiox_ingredients_count'
            AND pm.meta_value != ''
        ");
        
        return intval($count);
    }
    
    /**
     * Get last content update timestamp
     */
    public static function get_last_content_update() {
        global $wpdb;
        
        $last_update = $wpdb->get_var("
            SELECT MAX(p.post_modified)
            FROM {$wpdb->posts} p
            INNER JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id
            WHERE p.post_status = 'publish'
            AND p.post_type IN ('post', 'page')
            AND pm.meta_key = '_aiox_processed'
            AND pm.meta_value = '1'
        ");
        
        return $last_update ? date('c', strtotime($last_update)) : null;
    }
    
    /**
     * Get content statistics
     */
    public static function get_content_statistics() {
        return array(
            'total_posts' => self::get_total_processed_posts(),
            'total_qa_pairs' => self::get_total_qa_pairs(),
            'total_ingredients' => self::get_total_ingredients(),
            'last_updated' => self::get_last_content_update(),
            'ingredients_system_enabled' => get_option('aiox_enable_ingredients_system', true),
            'metadata_classification_enabled' => get_option('aiox_enable_metadata_classification', true),
            'legacy_qa_compatibility' => get_option('aiox_enable_legacy_qa_compatibility', true)
        );
    }
}
