<?php
/**
 * Safe cleanup methods for Enhanced Deletion
 * These methods provide fallback functionality to prevent site crashes
 */

if (!defined('ABSPATH')) {
    exit;
}

trait AIOX_Enhanced_Deletion_Safe {
    
    /**
     * Perform basic cleanup without comprehensive data
     */
    private function perform_basic_cleanup($post_id) {
        try {
            AIOX_Logger::debug("Enhanced Deletion: Performing basic cleanup for post {$post_id}");
            
            // Clean up basic post meta
            $meta_keys = array(
                '_aiox_processed_data',
                '_aiox_processing_complete',
                '_aiox_last_processed',
                '_aiox_ingredients',
                '_aiox_metadata',
                '_aiox_capsule_id',
                '_aiox_badge_updated',
                '_aiox_sync_version',
                '_aiox_last_error'
            );
            
            foreach ($meta_keys as $key) {
                delete_post_meta($post_id, $key);
            }
            
            // Try to clean up manifesto file if it exists
            $well_known_dir = ABSPATH . '.well-known';
            $manifesto_file = $well_known_dir . '/ai-info-' . $post_id . '.json';
            if (file_exists($manifesto_file)) {
                unlink($manifesto_file);
                AIOX_Logger::debug("Enhanced Deletion: Removed manifesto file for post {$post_id}");
            }
            
            AIOX_Logger::debug("Enhanced Deletion: Basic cleanup completed for post {$post_id}");
            
        } catch (Exception $e) {
            AIOX_Logger::error("Enhanced Deletion: Basic cleanup failed for post {$post_id}", array(
                'error' => $e->getMessage()
            ));
        }
    }
    
    /**
     * Safe comprehensive cleanup with error handling
     */
    private function perform_comprehensive_cleanup_safe($post_id, $aiox_data) {
        $cleanup_result = array(
            'success' => true,
            'components_cleaned' => array(),
            'files_removed' => array(),
            'errors' => array()
        );
        
        try {
            // Try comprehensive cleanup first
            return $this->perform_comprehensive_cleanup($post_id, $aiox_data);
            
        } catch (Exception $e) {
            AIOX_Logger::warning("Enhanced Deletion: Comprehensive cleanup failed, falling back to basic cleanup for post {$post_id}", array(
                'error' => $e->getMessage()
            ));
            
            // Fall back to basic cleanup
            $this->perform_basic_cleanup($post_id);
            
            $cleanup_result['success'] = false;
            $cleanup_result['errors'][] = 'Comprehensive cleanup failed, used basic cleanup: ' . $e->getMessage();
            
            return $cleanup_result;
        }
    }
    
    /**
     * Safe deletion statistics update
     */
    private function update_deletion_stats_safe($post_id, $cleanup_result) {
        try {
            $this->update_deletion_stats($post_id, $cleanup_result);
        } catch (Exception $e) {
            AIOX_Logger::error("Enhanced Deletion: Failed to update deletion stats for post {$post_id}", array(
                'error' => $e->getMessage()
            ));
            
            // Try basic stats update
            try {
                if (isset($this->deletion_stats) && is_array($this->deletion_stats)) {
                    $this->deletion_stats['posts_deleted']++;
                    $this->deletion_stats['last_deletion'] = current_time('mysql');
                    update_option('aiox_deletion_stats', $this->deletion_stats);
                }
            } catch (Exception $basic_e) {
                AIOX_Logger::error("Enhanced Deletion: Even basic stats update failed for post {$post_id}");
            }
        }
    }
}
