<?php
/**
 * AIOX Enhanced Deletion Handler
 * 
 * Extends the existing post deletion functionality to ensure comprehensive
 * cleanup of all AIOX components when posts are deleted.
 * 
 * Phase 1: Foundation setup - establishes enhanced deletion framework
 * without modifying existing deletion behavior.
 */

if (!defined('ABSPATH')) {
    exit;
}

// Ensure required classes are loaded
require_once AIOX_PLUGIN_DIR . 'core/class-aiox-comprehensive-sync.php';

class AIOX_Enhanced_Deletion {
    
    /**
     * Comprehensive sync handler
     */
    private $comprehensive_sync;
    
    /**
     * Deletion statistics
     */
    private $deletion_stats;
    
    /**
     * Constructor
     */
    public function __construct() {
        AIOX_Logger::info('AIOX Enhanced Deletion: Initializing Phase 1 foundation');
        
        try {
            // Initialize comprehensive sync handler
            $this->comprehensive_sync = new AIOX_Comprehensive_Sync();
            
            // Initialize deletion statistics (load existing or create new)
            $this->deletion_stats = get_option('aiox_deletion_stats', array(
                'posts_deleted' => 0,
                'components_cleaned' => 0,
                'files_removed' => 0,
                'last_deletion' => null,
                'errors' => array()
            ));
            
            // Register deletion hooks
            $this->register_deletion_hooks();
            
            AIOX_Logger::info('AIOX Enhanced Deletion: Phase 1 foundation initialized successfully');
            
        } catch (Exception $e) {
            AIOX_Logger::error('AIOX Enhanced Deletion: Failed to initialize Phase 1 foundation', array(
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ));
        }
    }
    
    /**
     * Register deletion hooks
     */
    private function register_deletion_hooks() {
        if (!function_exists('add_action')) {
            return;
        }
        
        // Post deletion hooks
        add_action('before_delete_post', array($this, 'on_before_delete_post'), 10);
        add_action('delete_post', array($this, 'on_delete_post'), 20); // Higher priority than existing
        add_action('deleted_post', array($this, 'on_after_delete_post'), 10);
        
        // Bulk deletion hooks
        add_action('aiox_bulk_delete_posts', array($this, 'on_bulk_delete_posts'));
        
        // Cleanup hooks
        add_action('aiox_cleanup_orphaned_data', array($this, 'cleanup_orphaned_data'));
        
        AIOX_Logger::debug('AIOX Enhanced Deletion: Deletion hooks registered');
    }
    
    /**
     * Handler for before post deletion
     * Captures post data before it's deleted for comprehensive cleanup
     */
    public function on_before_delete_post($post_id) {
        try {
            $post = get_post($post_id);
            if (!$post) {
                return;
            }
            
            // Skip if not a post type we handle
            if (!in_array($post->post_type, array('post', 'page'))) {
                return;
            }
            
            AIOX_Logger::info("Enhanced Deletion: Preparing for deletion of post {$post_id} ({$post->post_title})");
            
            // Capture current AIOX data for cleanup
            $aiox_data = $this->capture_aiox_data($post_id);
            
            // Store data temporarily for use in deletion handlers
            set_transient('aiox_deletion_data_' . $post_id, $aiox_data, 300); // 5 minutes
            
            AIOX_Logger::debug("Enhanced Deletion: Captured AIOX data for post {$post_id}", $aiox_data);
            
        } catch (Exception $e) {
            AIOX_Logger::error("Enhanced Deletion: Failed to prepare for deletion of post {$post_id}", array(
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ));
        }
    }
    
    /**
     * Handler for post deletion
     * Performs comprehensive cleanup of all AIOX components
     */
    public function on_delete_post($post_id) {
        // CRITICAL: Wrap everything in try-catch to prevent site crashes
        try {
            // Skip if comprehensive sync is not available
            if (!$this->comprehensive_sync) {
                AIOX_Logger::debug("Enhanced Deletion: Comprehensive sync not available, skipping cleanup for post {$post_id}");
                return;
            }
            
            // Get captured data
            $aiox_data = get_transient('aiox_deletion_data_' . $post_id);
            
            if (!$aiox_data) {
                AIOX_Logger::debug("Enhanced Deletion: No AIOX data found for post {$post_id}, performing basic cleanup");
                // Perform basic cleanup without comprehensive data
                $this->perform_basic_cleanup($post_id);
                return;
            }
            
            AIOX_Logger::info("Enhanced Deletion: Starting comprehensive cleanup for deleted post {$post_id}");
            
            // Perform comprehensive cleanup with additional error handling
            $cleanup_result = $this->perform_comprehensive_cleanup_safe($post_id, $aiox_data);
            
            // Update deletion statistics (with error handling)
            $this->update_deletion_stats_safe($post_id, $cleanup_result);
            
            // Clean up temporary data
            delete_transient('aiox_deletion_data_' . $post_id);
            
            AIOX_Logger::info("Enhanced Deletion: Completed comprehensive cleanup for deleted post {$post_id}", $cleanup_result);
            
        } catch (Exception $e) {
            // CRITICAL: Log error but don't re-throw to prevent site crash
            AIOX_Logger::error("Enhanced Deletion: Failed comprehensive cleanup for deleted post {$post_id}", array(
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ));
            
            // Attempt basic cleanup as fallback
            try {
                $this->perform_basic_cleanup($post_id);
                AIOX_Logger::info("Enhanced Deletion: Fallback basic cleanup completed for post {$post_id}");
            } catch (Exception $fallback_e) {
                AIOX_Logger::error("Enhanced Deletion: Even fallback cleanup failed for post {$post_id}", array(
                    'error' => $fallback_e->getMessage()
                ));
            }
        }
    }
    
    /**
     * Handler for after post deletion
     * Final cleanup and verification
     */
    public function on_after_delete_post($post_id) {
        try {
            AIOX_Logger::debug("Enhanced Deletion: Performing final cleanup verification for post {$post_id}");
            
            // Verify all AIOX data has been cleaned up
            $remaining_data = $this->check_remaining_aiox_data($post_id);
            
            if (!empty($remaining_data)) {
                AIOX_Logger::warning("Enhanced Deletion: Found remaining AIOX data for deleted post {$post_id}", $remaining_data);
                
                // Force cleanup of remaining data
                $this->force_cleanup_remaining_data($post_id, $remaining_data);
            } else {
                AIOX_Logger::debug("Enhanced Deletion: Verified complete cleanup for post {$post_id}");
            }
            
        } catch (Exception $e) {
            AIOX_Logger::error("Enhanced Deletion: Failed final cleanup verification for post {$post_id}", array(
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ));
        }
    }
    
    /**
     * Handler for bulk post deletion
     */
    public function on_bulk_delete_posts($post_ids) {
        try {
            AIOX_Logger::info("Enhanced Deletion: Starting bulk deletion cleanup for " . count($post_ids) . " posts");
            
            $bulk_results = array();
            
            foreach ($post_ids as $post_id) {
                try {
                    // Capture data
                    $aiox_data = $this->capture_aiox_data($post_id);
                    
                    // Perform cleanup
                    $cleanup_result = $this->perform_comprehensive_cleanup($post_id, $aiox_data);
                    
                    $bulk_results[$post_id] = $cleanup_result;
                    
                } catch (Exception $e) {
                    $bulk_results[$post_id] = array(
                        'success' => false,
                        'error' => $e->getMessage()
                    );
                }
            }
            
            AIOX_Logger::info("Enhanced Deletion: Completed bulk deletion cleanup", $bulk_results);
            
            return $bulk_results;
            
        } catch (Exception $e) {
            AIOX_Logger::error("Enhanced Deletion: Failed bulk deletion cleanup", array(
                'post_ids' => $post_ids,
                'error' => $e->getMessage()
            ));
            
            throw $e;
        }
    }
    
    /**
     * Capture AIOX data before deletion
     */
    private function capture_aiox_data($post_id) {
        $aiox_data = array(
            'post_id' => $post_id,
            'post_meta' => array(),
            'capsule_id' => null,
            'has_ingredients' => false,
            'has_metadata' => false,
            'has_processed_data' => false,
            'manifesto_files' => array()
        );
        
        // Capture all AIOX post meta
        $meta_keys = array(
            '_aiox_processed_data',
            '_aiox_processing_complete',
            '_aiox_last_processed',
            '_aiox_ingredients',
            '_aiox_metadata',
            '_aiox_capsule_id',
            '_aiox_badge_updated',
            '_aiox_sync_version',
            '_aiox_last_error'
        );
        
        foreach ($meta_keys as $key) {
            $value = get_post_meta($post_id, $key, true);
            if (!empty($value)) {
                $aiox_data['post_meta'][$key] = $value;
            }
        }
        
        // Capture specific component data
        $aiox_data['capsule_id'] = get_post_meta($post_id, '_aiox_capsule_id', true);
        $aiox_data['has_ingredients'] = !empty(get_post_meta($post_id, '_aiox_ingredients', true));
        $aiox_data['has_metadata'] = !empty(get_post_meta($post_id, '_aiox_metadata', true));
        $aiox_data['has_processed_data'] = !empty(get_post_meta($post_id, '_aiox_processed_data', true));
        
        // Check for manifesto files
        $well_known_dir = ABSPATH . '.well-known';
        $ai_info_file = $well_known_dir . '/ai-info-' . $post_id . '.json';
        if (file_exists($ai_info_file)) {
            $aiox_data['manifesto_files'][] = 'ai-info-' . $post_id . '.json';
        }
        
        return $aiox_data;
    }
    
    /**
     * Perform comprehensive cleanup
     */
    private function perform_comprehensive_cleanup($post_id, $aiox_data) {
        $cleanup_result = array(
            'post_id' => $post_id,
            'success' => true,
            'components_cleaned' => array(),
            'files_removed' => array(),
            'errors' => array()
        );
        
        try {
            // 1. Clean up capsule
            if (!empty($aiox_data['capsule_id'])) {
                $capsule_result = $this->cleanup_capsule($post_id, $aiox_data['capsule_id']);
                $cleanup_result['components_cleaned']['capsule'] = $capsule_result;
            }
            
            // 2. Clean up ingredients
            if ($aiox_data['has_ingredients']) {
                $ingredients_result = $this->cleanup_ingredients($post_id);
                $cleanup_result['components_cleaned']['ingredients'] = $ingredients_result;
            }
            
            // 3. Clean up metadata
            if ($aiox_data['has_metadata']) {
                $metadata_result = $this->cleanup_metadata($post_id);
                $cleanup_result['components_cleaned']['metadata'] = $metadata_result;
            }
            
            // 4. Clean up manifesto files
            if (!empty($aiox_data['manifesto_files'])) {
                $manifesto_result = $this->cleanup_manifesto_files($post_id, $aiox_data['manifesto_files']);
                $cleanup_result['components_cleaned']['manifesto'] = $manifesto_result;
                $cleanup_result['files_removed'] = array_merge($cleanup_result['files_removed'], isset($manifesto_result['files_removed']) ? $manifesto_result['files_removed'] : array());
            }
            
            // 5. Clean up badge data
            if ($aiox_data['has_processed_data']) {
                $badge_result = $this->cleanup_badge_data($post_id);
                $cleanup_result['components_cleaned']['badge'] = $badge_result;
            }
            
            // 6. Clean up all post meta
            $meta_result = $this->cleanup_post_meta($post_id, $aiox_data['post_meta']);
            $cleanup_result['components_cleaned']['post_meta'] = $meta_result;
            
            // 7. Update main manifesto (remove post reference)
            $this->update_main_manifesto_on_deletion($post_id);
            
        } catch (Exception $e) {
            $cleanup_result['success'] = false;
            $cleanup_result['errors'][] = $e->getMessage();
        }
        
        return $cleanup_result;
    }
    
    /**
     * Clean up capsule component
     */
    private function cleanup_capsule($post_id, $capsule_id) {
        try {
            // Use comprehensive processor to delete capsule
            $this->comprehensive_sync->sync_post_components($post_id, 'delete');
            
            return array('action' => 'deleted', 'capsule_id' => $capsule_id);
            
        } catch (Exception $e) {
            AIOX_Logger::warning("Enhanced Deletion: Failed to cleanup capsule {$capsule_id} for post {$post_id}: " . $e->getMessage());
            return array('action' => 'failed', 'error' => $e->getMessage());
        }
    }
    
    /**
     * Clean up ingredients component
     */
    private function cleanup_ingredients($post_id) {
        try {
            AIOX_Ingredients::delete_post_ingredients($post_id);
            
            return array('action' => 'deleted');
            
        } catch (Exception $e) {
            AIOX_Logger::warning("Enhanced Deletion: Failed to cleanup ingredients for post {$post_id}: " . $e->getMessage());
            return array('action' => 'failed', 'error' => $e->getMessage());
        }
    }
    
    /**
     * Clean up metadata component
     */
    private function cleanup_metadata($post_id) {
        try {
            AIOX_Metadata::delete_post_metadata($post_id);
            
            return array('action' => 'deleted');
            
        } catch (Exception $e) {
            AIOX_Logger::warning("Enhanced Deletion: Failed to cleanup metadata for post {$post_id}: " . $e->getMessage());
            return array('action' => 'failed', 'error' => $e->getMessage());
        }
    }
    
    /**
     * Clean up manifesto files
     */
    private function cleanup_manifesto_files($post_id, $manifesto_files) {
        $result = array(
            'action' => 'cleaned',
            'files_removed' => array(),
            'errors' => array()
        );
        
        $well_known_dir = ABSPATH . '.well-known';
        
        foreach ($manifesto_files as $file) {
            try {
                $file_path = $well_known_dir . '/' . $file;
                if (file_exists($file_path)) {
                    if (unlink($file_path)) {
                        $result['files_removed'][] = $file;
                        AIOX_Logger::debug("Enhanced Deletion: Removed manifesto file {$file} for post {$post_id}");
                    } else {
                        $result['errors'][] = "Failed to remove file: {$file}";
                    }
                }
            } catch (Exception $e) {
                $result['errors'][] = "Error removing file {$file}: " . $e->getMessage();
            }
        }
        
        return $result;
    }
    
    /**
     * Clean up badge data
     */
    private function cleanup_badge_data($post_id) {
        try {
            delete_post_meta($post_id, '_aiox_processed_data');
            delete_post_meta($post_id, '_aiox_badge_updated');
            
            return array('action' => 'cleared');
            
        } catch (Exception $e) {
            AIOX_Logger::warning("Enhanced Deletion: Failed to cleanup badge data for post {$post_id}: " . $e->getMessage());
            return array('action' => 'failed', 'error' => $e->getMessage());
        }
    }
    
    /**
     * Clean up all AIOX post meta
     */
    private function cleanup_post_meta($post_id, $post_meta) {
        $result = array(
            'action' => 'cleaned',
            'meta_removed' => array(),
            'errors' => array()
        );
        
        foreach ($post_meta as $key => $value) {
            try {
                if (delete_post_meta($post_id, $key)) {
                    $result['meta_removed'][] = $key;
                } else {
                    $result['errors'][] = "Failed to remove meta: {$key}";
                }
            } catch (Exception $e) {
                $result['errors'][] = "Error removing meta {$key}: " . $e->getMessage();
            }
        }
        
        return $result;
    }
    
    /**
     * Update main manifesto on deletion
     */
    private function update_main_manifesto_on_deletion($post_id) {
        try {
            // This would use the comprehensive processor to update the main manifest
            // removing references to the deleted post
            if (method_exists($this->comprehensive_processor, 'remove_from_main_manifest')) {
                $this->comprehensive_processor->remove_from_main_manifest($post_id);
            }
            
            AIOX_Logger::debug("Enhanced Deletion: Updated main manifesto for deleted post {$post_id}");
            
        } catch (Exception $e) {
            AIOX_Logger::warning("Enhanced Deletion: Failed to update main manifesto for deleted post {$post_id}: " . $e->getMessage());
        }
    }
    
    /**
     * Check for remaining AIOX data after deletion
     */
    private function check_remaining_aiox_data($post_id) {
        $remaining_data = array();
        
        // Check for remaining post meta
        global $wpdb;
        $remaining_meta = $wpdb->get_results($wpdb->prepare("
            SELECT meta_key, meta_value 
            FROM {$wpdb->postmeta} 
            WHERE post_id = %d 
            AND meta_key LIKE '_aiox_%'
        ", $post_id));
        
        if (!empty($remaining_meta)) {
            $remaining_data['post_meta'] = $remaining_meta;
        }
        
        // Check for remaining files
        $well_known_dir = ABSPATH . '.well-known';
        $ai_info_file = $well_known_dir . '/ai-info-' . $post_id . '.json';
        if (file_exists($ai_info_file)) {
            $remaining_data['files'][] = 'ai-info-' . $post_id . '.json';
        }
        
        return $remaining_data;
    }
    
    /**
     * Force cleanup of remaining data
     */
    private function force_cleanup_remaining_data($post_id, $remaining_data) {
        // Force remove remaining post meta
        if (isset($remaining_data['post_meta'])) {
            foreach ($remaining_data['post_meta'] as $meta) {
                delete_post_meta($post_id, $meta->meta_key);
                AIOX_Logger::debug("Enhanced Deletion: Force removed remaining meta {$meta->meta_key} for post {$post_id}");
            }
        }
        
        // Force remove remaining files
        if (isset($remaining_data['files'])) {
            $well_known_dir = ABSPATH . '.well-known';
            foreach ($remaining_data['files'] as $file) {
                $file_path = $well_known_dir . '/' . $file;
                if (file_exists($file_path)) {
                    unlink($file_path);
                    AIOX_Logger::debug("Enhanced Deletion: Force removed remaining file {$file} for post {$post_id}");
                }
            }
        }
    }
    
    /**
     * Update deletion statistics
     */
    private function update_deletion_stats($post_id, $cleanup_result) {
        $this->deletion_stats['posts_deleted']++;
        $this->deletion_stats['components_cleaned'] += count(isset($cleanup_result['components_cleaned']) ? $cleanup_result['components_cleaned'] : array());
        $this->deletion_stats['files_removed'] += count(isset($cleanup_result['files_removed']) ? $cleanup_result['files_removed'] : array());
        $this->deletion_stats['last_deletion'] = current_time('mysql');
        
        if (!$cleanup_result['success']) {
            $this->deletion_stats['errors'] = array_merge($this->deletion_stats['errors'], isset($cleanup_result['errors']) ? $cleanup_result['errors'] : array());
        }
        
        // Store stats in options for persistence
        update_option('aiox_deletion_stats', $this->deletion_stats);
    }
    
    /**
     * Cleanup orphaned data
     */
    public function cleanup_orphaned_data() {
        try {
            AIOX_Logger::info("Enhanced Deletion: Starting orphaned data cleanup");
            
            global $wpdb;
            
            // Find orphaned post meta
            $orphaned_meta = $wpdb->get_results("
                SELECT pm.post_id, pm.meta_key 
                FROM {$wpdb->postmeta} pm 
                LEFT JOIN {$wpdb->posts} p ON pm.post_id = p.ID 
                WHERE pm.meta_key LIKE '_aiox_%' 
                AND p.ID IS NULL
            ");
            
            foreach ($orphaned_meta as $meta) {
                delete_post_meta($meta->post_id, $meta->meta_key);
            }
            
            // Find orphaned files
            $well_known_dir = ABSPATH . '.well-known';
            if (is_dir($well_known_dir)) {
                $files = glob($well_known_dir . '/ai-info-*.json');
                foreach ($files as $file) {
                    if (preg_match('/ai-info-(\d+)\.json$/', basename($file), $matches)) {
                        $post_id = intval($matches[1]);
                        if (!get_post($post_id)) {
                            unlink($file);
                            AIOX_Logger::debug("Enhanced Deletion: Removed orphaned file " . basename($file));
                        }
                    }
                }
            }
            
            AIOX_Logger::info("Enhanced Deletion: Completed orphaned data cleanup", array(
                'orphaned_meta' => count($orphaned_meta),
                'orphaned_files' => count(isset($files) ? $files : array())
            ));
            
        } catch (Exception $e) {
            AIOX_Logger::error("Enhanced Deletion: Failed orphaned data cleanup", array(
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ));
        }
    }
    
    /**
     * Get deletion statistics
     */
    public function get_deletion_stats() {
        return get_option('aiox_deletion_stats', $this->deletion_stats);
    }
    
    /**
     * Reset deletion statistics
     */
    public function reset_deletion_stats() {
        $this->deletion_stats = array(
            'posts_deleted' => 0,
            'components_cleaned' => 0,
            'files_removed' => 0,
            'last_deletion' => null,
            'errors' => array()
        );
        
        update_option('aiox_deletion_stats', $this->deletion_stats);
    }
    
    /**
     * Perform basic cleanup without comprehensive data
     */
    private function perform_basic_cleanup($post_id) {
        try {
            AIOX_Logger::debug("Enhanced Deletion: Performing basic cleanup for post {$post_id}");
            
            // Clean up basic post meta
            $meta_keys = array(
                '_aiox_processed_data',
                '_aiox_processing_complete',
                '_aiox_last_processed',
                '_aiox_ingredients',
                '_aiox_metadata',
                '_aiox_capsule_id',
                '_aiox_badge_updated',
                '_aiox_sync_version',
                '_aiox_last_error'
            );
            
            foreach ($meta_keys as $key) {
                delete_post_meta($post_id, $key);
            }
            
            // Try to clean up manifesto file if it exists
            $well_known_dir = ABSPATH . '.well-known';
            $manifesto_file = $well_known_dir . '/ai-info-' . $post_id . '.json';
            if (file_exists($manifesto_file)) {
                unlink($manifesto_file);
                AIOX_Logger::debug("Enhanced Deletion: Removed manifesto file for post {$post_id}");
            }
            
            AIOX_Logger::debug("Enhanced Deletion: Basic cleanup completed for post {$post_id}");
            
        } catch (Exception $e) {
            AIOX_Logger::error("Enhanced Deletion: Basic cleanup failed for post {$post_id}", array(
                'error' => $e->getMessage()
            ));
        }
    }
    
    /**
     * Safe comprehensive cleanup with error handling
     */
    private function perform_comprehensive_cleanup_safe($post_id, $aiox_data) {
        $cleanup_result = array(
            'success' => true,
            'components_cleaned' => array(),
            'files_removed' => array(),
            'errors' => array()
        );
        
        try {
            // Try comprehensive cleanup first
            return $this->perform_comprehensive_cleanup($post_id, $aiox_data);
            
        } catch (Exception $e) {
            AIOX_Logger::warning("Enhanced Deletion: Comprehensive cleanup failed, falling back to basic cleanup for post {$post_id}", array(
                'error' => $e->getMessage()
            ));
            
            // Fall back to basic cleanup
            $this->perform_basic_cleanup($post_id);
            
            $cleanup_result['success'] = false;
            $cleanup_result['errors'][] = 'Comprehensive cleanup failed, used basic cleanup: ' . $e->getMessage();
            
            return $cleanup_result;
        }
    }
    
    /**
     * Safe deletion statistics update
     */
    private function update_deletion_stats_safe($post_id, $cleanup_result) {
        try {
            $this->update_deletion_stats($post_id, $cleanup_result);
        } catch (Exception $e) {
            AIOX_Logger::error("Enhanced Deletion: Failed to update deletion stats for post {$post_id}", array(
                'error' => $e->getMessage()
            ));
            
            // Try basic stats update
            try {
                if (isset($this->deletion_stats) && is_array($this->deletion_stats)) {
                    $this->deletion_stats['posts_deleted']++;
                    $this->deletion_stats['last_deletion'] = current_time('mysql');
                    update_option('aiox_deletion_stats', $this->deletion_stats);
                }
            } catch (Exception $basic_e) {
                AIOX_Logger::error("Enhanced Deletion: Even basic stats update failed for post {$post_id}");
            }
        }
    }
}
