<?php
/**
 * AIOX Enhanced Hooks
 * 
 * Extends the existing AIOX_Hooks class to add comprehensive processing
 * capabilities while preserving all existing hook functionality.
 * 
 * Phase 1: Foundation setup - establishes enhanced hook framework
 * without modifying existing hook behavior.
 */

if (!defined('ABSPATH')) {
    exit;
}

// Ensure the parent class is loaded
require_once AIOX_PLUGIN_DIR . 'includes/class-aiox-hooks.php';
require_once AIOX_PLUGIN_DIR . 'core/class-aiox-comprehensive-sync.php';

class AIOX_Enhanced_Hooks extends AIOX_Hooks {
    
    /**
     * Comprehensive sync handler
     */
    private $comprehensive_sync;
    
    /**
     * Hook statistics
     */
    private $hook_stats;
    
    /**
     * Constructor
     */
    public function __construct() {
        AIOX_Logger::info('AIOX Enhanced Hooks: Initializing Phase 1 foundation');
        
        // CRITICAL: Call parent constructor first to preserve all existing functionality
        parent::__construct();
        
        try {
            // Initialize comprehensive sync handler
            $this->comprehensive_sync = new AIOX_Comprehensive_Sync();
            
            // Initialize hook statistics (load existing or create new)
            $this->hook_stats = get_option('aiox_enhanced_hook_stats', array(
                'posts_processed' => 0,
                'posts_deleted' => 0,
                'comprehensive_processes' => 0,
                'last_activity' => current_time('mysql')
            ));
            
            // Register enhanced hooks (additive - don't remove existing ones)
            $this->register_enhanced_hooks();
            
            AIOX_Logger::info('AIOX Enhanced Hooks: Phase 1 foundation initialized successfully');
            
        } catch (Exception $e) {
            AIOX_Logger::error('AIOX Enhanced Hooks: Failed to initialize Phase 1 foundation', array(
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ));
        }
    }
    
    /**
     * Register enhanced hooks
     * These are additional hooks that extend functionality without replacing existing ones
     */
    private function register_enhanced_hooks() {
        if (!function_exists('add_action')) {
            return;
        }
        
        // Enhanced post lifecycle hooks
        add_action('save_post', array($this, 'on_enhanced_post_save'), 20, 3); // Higher priority than parent
        add_action('delete_post', array($this, 'on_enhanced_post_delete'), 20); // Higher priority than parent
        add_action('transition_post_status', array($this, 'on_enhanced_post_status_change'), 20, 3);
        
        // Background processing hooks
        add_action('aiox_process_post_comprehensive', array($this, 'process_post_comprehensive_background'));
        add_action('aiox_sync_post_components', array($this, 'sync_post_components_background'));
        
        // Auto-processing retry hook
        add_action('aiox_auto_process_post_retry', array($this, 'process_post_retry_background'));
        
        // Cleanup hooks
        add_action('aiox_cleanup_comprehensive_data', array($this, 'cleanup_comprehensive_data'));
        
        AIOX_Logger::debug('AIOX Enhanced Hooks: Enhanced hooks registered');
    }
    
    /**
     * Enhanced post save handler
     * This extends the existing post save functionality with comprehensive processing
     */
    public function on_enhanced_post_save($post_id, $post, $update) {
        // IMPORTANT: Don't call parent::on_post_save() here as it's already called
        // by the original hook registration. This is an additional handler.
        
        try {
            // Handle auto-processing first
            $this->handle_auto_process_on_save($post_id, $post, $update);
            
            // Skip if this is not a comprehensive processing scenario
            if (!$this->should_process_comprehensively($post_id, $post)) {
                return;
            }
            
            AIOX_Logger::info("Enhanced Hooks: Starting comprehensive processing for post {$post_id}");
            
            // Check if comprehensive processing is enabled
            $comprehensive_enabled = get_option('aiox_enable_comprehensive_processing', true);
            
            if (!$comprehensive_enabled) {
                AIOX_Logger::debug("Enhanced Hooks: Comprehensive processing disabled, skipping post {$post_id}");
                return;
            }
            
            // Schedule comprehensive processing in background to avoid blocking post save
            wp_schedule_single_event(time() + 15, 'aiox_process_post_comprehensive', array($post_id));
            
            // Update hook statistics
            $this->hook_stats['posts_processed']++;
            $this->hook_stats['last_activity'] = current_time('mysql');
            update_option('aiox_enhanced_hook_stats', $this->hook_stats);
            
            AIOX_Logger::info("Enhanced Hooks: Scheduled comprehensive processing for post {$post_id}");
            
        } catch (Exception $e) {
            AIOX_Logger::error("Enhanced Hooks: Failed to process post save for post {$post_id}", array(
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ));
        }
    }
    
    /**
     * Handle auto-processing on post save
     */
    public function handle_auto_process_on_save($post_id, $post, $update) {
        // Check if auto-processing is enabled
        if (!get_option('aiox_auto_process_on_save', false)) {
            return;
        }
        
        // Skip if this is an autosave
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return;
        }
        
        // Skip if this is a revision
        if (wp_is_post_revision($post_id)) {
            return;
        }
        
        // Only process published posts and pages
        if (!in_array($post->post_type, array('post', 'page'))) {
            return;
        }
        
        if ($post->post_status !== 'publish') {
            return;
        }
        
        // Skip if user doesn't have permission
        if (!current_user_can('edit_post', $post_id)) {
            return;
        }
        
        // Skip if already processed recently (within last 5 minutes)
        $last_processed = get_post_meta($post_id, '_aiox_last_processed', true);
        if ($last_processed && (time() - $last_processed) < 300) {
            AIOX_Logger::debug("Auto-process skipped for post {$post_id}: recently processed");
            return;
        }
        
        AIOX_Logger::info("Auto-process triggered for post {$post_id}: {$post->post_title}");
        
        // Process immediately for better user experience
        $this->process_post_immediately($post_id);
    }
    
    /**
     * Process post immediately (synchronous)
     */
    private function process_post_immediately($post_id) {
        try {
            // Update last processed timestamp
            update_post_meta($post_id, '_aiox_last_processed', time());
            
            // Initialize comprehensive processor
            if (!class_exists('AIOX_Comprehensive_Processor')) {
                require_once AIOX_PLUGIN_DIR . 'core/class-aiox-comprehensive-processor.php';
            }
            
            $processor = new AIOX_Comprehensive_Processor();
            $result = $processor->process_post_completely($post_id);
            
            if ($result && is_array($result) && isset($result['success']) && $result['success']) {
                AIOX_Logger::info("Auto-processed post {$post_id} successfully");
                
                // Update processing status
                update_post_meta($post_id, '_aiox_auto_processed', true);
                update_post_meta($post_id, '_aiox_auto_processed_time', current_time('mysql'));
                
                // Add admin notice for immediate feedback
                if (is_admin()) {
                    add_action('admin_notices', function() use ($post_id) {
                        echo '<div class="notice notice-success is-dismissible">';
                        echo '<p><strong>AIOX:</strong> Post automatically processed with AI analysis and ingredients extraction.</p>';
                        echo '</div>';
                    });
                }
                
            } else {
                $error_message = isset($result['message']) ? $result['message'] : 'Unknown error';
                AIOX_Logger::warning("Auto-processing failed for post {$post_id}: {$error_message}");
                
                // Schedule retry in background
                wp_schedule_single_event(time() + 60, 'aiox_auto_process_post_retry', array($post_id));
            }
            
        } catch (Exception $e) {
            AIOX_Logger::error("Auto-processing exception for post {$post_id}: " . $e->getMessage());
            
            // Schedule retry in background
            wp_schedule_single_event(time() + 60, 'aiox_auto_process_post_retry', array($post_id));
        }
    }
    
    /**
     * Enhanced post deletion handler
     * This extends the existing post deletion functionality with comprehensive cleanup
     */
    public function on_enhanced_post_delete($post_id) {
        // IMPORTANT: Don't call parent::on_post_delete() here as it's already called
        // by the original hook registration. This is an additional handler.
        
        try {
            $post = get_post($post_id);
            if (!$post) {
                return;
            }
            
            // Skip attachments and other non-content post types
            // This prevents errors during plugin uploads which create temporary attachments
            $skip_types = array('attachment', 'revision', 'nav_menu_item', 'custom_css', 'customize_changeset', 'oembed_cache', 'user_request', 'wp_block', 'wp_template', 'wp_template_part', 'wp_global_styles', 'wp_navigation');
            if (in_array($post->post_type, $skip_types)) {
                return;
            }
            
            // Only process post types we care about
            $allowed_types = get_option('aiox_allowed_post_types', array('post', 'page'));
            if (!in_array($post->post_type, $allowed_types)) {
                return;
            }
            
            AIOX_Logger::info("Enhanced Hooks: Starting comprehensive cleanup for deleted post {$post_id}");
            
            // Perform comprehensive cleanup
            $sync_result = $this->comprehensive_sync->sync_post_components($post_id, 'delete');
            
            // Update hook statistics
            $this->hook_stats['posts_deleted']++;
            $this->hook_stats['last_activity'] = current_time('mysql');
            update_option('aiox_enhanced_hook_stats', $this->hook_stats);
            
            AIOX_Logger::info("Enhanced Hooks: Completed comprehensive cleanup for deleted post {$post_id}", $sync_result);
            
        } catch (Exception $e) {
            AIOX_Logger::error("Enhanced Hooks: Failed to process post deletion for post {$post_id}", array(
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ));
        }
    }
    
    /**
     * Enhanced post status change handler
     * This extends the existing post status change functionality
     */
    public function on_enhanced_post_status_change($new_status, $old_status, $post) {
        try {
            // Skip if not relevant for comprehensive processing
            if (!$this->should_process_comprehensively($post->ID, $post)) {
                return;
            }
            
            AIOX_Logger::debug("Enhanced Hooks: Post status changed for post {$post->ID}: {$old_status} -> {$new_status}");
            
            // Handle status changes that affect comprehensive processing
            if ($new_status === 'publish' && $old_status !== 'publish') {
                // Post was published - schedule comprehensive processing
                $comprehensive_enabled = get_option('aiox_enable_comprehensive_processing', true);
                if ($comprehensive_enabled) {
                    wp_schedule_single_event(time() + 20, 'aiox_process_post_comprehensive', array($post->ID));
                    AIOX_Logger::info("Enhanced Hooks: Scheduled comprehensive processing for newly published post {$post->ID}");
                }
            } elseif ($old_status === 'publish' && $new_status !== 'publish') {
                // Post was unpublished - clean up comprehensive data
                wp_schedule_single_event(time() + 5, 'aiox_sync_post_components', array($post->ID, 'delete'));
                AIOX_Logger::info("Enhanced Hooks: Scheduled comprehensive cleanup for unpublished post {$post->ID}");
            }
            
        } catch (Exception $e) {
            AIOX_Logger::error("Enhanced Hooks: Failed to process status change for post {$post->ID}", array(
                'old_status' => $old_status,
                'new_status' => $new_status,
                'error' => $e->getMessage()
            ));
        }
    }
    
    /**
     * Background comprehensive processing handler
     */
    public function process_post_comprehensive_background($post_id) {
        try {
            AIOX_Logger::info("Enhanced Hooks: Starting background comprehensive processing for post {$post_id}");
            
            // Perform comprehensive synchronization
            $sync_result = $this->comprehensive_sync->sync_post_components($post_id, 'update');
            
            // Update statistics
            $this->hook_stats['comprehensive_processes']++;
            $this->hook_stats['last_activity'] = current_time('mysql');
            update_option('aiox_enhanced_hook_stats', $this->hook_stats);
            
            if ($sync_result['success']) {
                AIOX_Logger::info("Enhanced Hooks: Completed background comprehensive processing for post {$post_id}", $sync_result);
            } else {
                AIOX_Logger::warning("Enhanced Hooks: Background comprehensive processing had errors for post {$post_id}", $sync_result);
            }
            
        } catch (Exception $e) {
            AIOX_Logger::error("Enhanced Hooks: Background comprehensive processing failed for post {$post_id}", array(
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ));
        }
    }
    
    /**
     * Background processing for auto-processing retries
     */
    public function process_post_retry_background($post_id) {
        try {
            // Check if post still exists and is published
            $post = get_post($post_id);
            if (!$post || $post->post_status !== 'publish') {
                return;
            }
            
            AIOX_Logger::info("Retrying auto-processing for post {$post_id}");
            
            // Initialize comprehensive processor
            if (!class_exists('AIOX_Comprehensive_Processor')) {
                require_once AIOX_PLUGIN_DIR . 'core/class-aiox-comprehensive-processor.php';
            }
            
            $processor = new AIOX_Comprehensive_Processor();
            $result = $processor->process_post_completely($post_id);
            
            if ($result && is_array($result) && isset($result['success']) && $result['success']) {
                AIOX_Logger::info("Background auto-processed post {$post_id} successfully");
                
                // Update processing status
                update_post_meta($post_id, '_aiox_auto_processed', true);
                update_post_meta($post_id, '_aiox_auto_processed_time', current_time('mysql'));
                
            } else {
                $error_message = isset($result['message']) ? $result['message'] : 'Unknown error';
                AIOX_Logger::error("Background auto-processing retry failed for post {$post_id}: {$error_message}");
            }
            
        } catch (Exception $e) {
            AIOX_Logger::error("Background auto-processing retry exception for post {$post_id}: " . $e->getMessage());
        }
    }
    
    /**
     * Background component synchronization handler
     */
    public function sync_post_components_background($post_id, $action = 'update') {
        try {
            AIOX_Logger::info("Enhanced Hooks: Starting background component sync for post {$post_id} (action: {$action})");
            
            // Perform component synchronization
            $sync_result = $this->comprehensive_sync->sync_post_components($post_id, $action);
            
            if ($sync_result['success']) {
                AIOX_Logger::info("Enhanced Hooks: Completed background component sync for post {$post_id}", $sync_result);
            } else {
                AIOX_Logger::warning("Enhanced Hooks: Background component sync had errors for post {$post_id}", $sync_result);
            }
            
        } catch (Exception $e) {
            AIOX_Logger::error("Enhanced Hooks: Background component sync failed for post {$post_id}", array(
                'action' => $action,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ));
        }
    }
    
    /**
     * Cleanup comprehensive data handler
     */
    public function cleanup_comprehensive_data() {
        try {
            AIOX_Logger::info("Enhanced Hooks: Starting comprehensive data cleanup");
            
            // Clean up orphaned data
            $this->cleanup_orphaned_components();
            
            // Clean up old processing logs
            $this->cleanup_old_processing_logs();
            
            // Reset error counters
            $this->reset_error_counters();
            
            AIOX_Logger::info("Enhanced Hooks: Completed comprehensive data cleanup");
            
        } catch (Exception $e) {
            AIOX_Logger::error("Enhanced Hooks: Comprehensive data cleanup failed", array(
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ));
        }
    }
    
    /**
     * Check if post should be processed comprehensively
     */
    private function should_process_comprehensively($post_id, $post) {
        // Skip autosaves
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return false;
        }
        
        // Skip revisions
        if (wp_is_post_revision($post_id)) {
            return false;
        }
        
        // Only process posts and pages
        if (!in_array($post->post_type, array('post', 'page'))) {
            return false;
        }
        
        // Only process published posts
        if ($post->post_status !== 'publish') {
            return false;
        }
        
        // Check user permissions
        if (!current_user_can('edit_post', $post_id)) {
            return false;
        }
        
        return true;
    }
    
    /**
     * Clean up orphaned components
     */
    private function cleanup_orphaned_components() {
        global $wpdb;
        
        // Find posts with AIOX data but no longer exist
        $orphaned_meta = $wpdb->get_results("
            SELECT pm.post_id, pm.meta_key 
            FROM {$wpdb->postmeta} pm 
            LEFT JOIN {$wpdb->posts} p ON pm.post_id = p.ID 
            WHERE pm.meta_key LIKE '_aiox_%' 
            AND p.ID IS NULL
        ");
        
        foreach ($orphaned_meta as $meta) {
            delete_post_meta($meta->post_id, $meta->meta_key);
            AIOX_Logger::debug("Enhanced Hooks: Cleaned up orphaned meta {$meta->meta_key} for post {$meta->post_id}");
        }
        
        AIOX_Logger::info("Enhanced Hooks: Cleaned up " . count($orphaned_meta) . " orphaned meta entries");
    }
    
    /**
     * Clean up old processing logs
     */
    private function cleanup_old_processing_logs() {
        // Clean up logs older than 30 days
        $cutoff_date = date('Y-m-d H:i:s', strtotime('-30 days'));
        
        // This would integrate with the existing AIOX_Logger cleanup if available
        if (class_exists('AIOX_Logger') && method_exists('AIOX_Logger', 'cleanup_old_logs')) {
            AIOX_Logger::cleanup_old_logs($cutoff_date);
        }
    }
    
    /**
     * Reset error counters
     */
    private function reset_error_counters() {
        // Reset sync statistics errors
        $sync_stats = $this->comprehensive_sync->get_sync_stats();
        $sync_stats['errors'] = array();
        update_option('aiox_sync_stats', $sync_stats);
        
        AIOX_Logger::debug("Enhanced Hooks: Reset error counters");
    }
    
    /**
     * Get enhanced hook statistics
     */
    public function get_hook_stats() {
        return get_option('aiox_enhanced_hook_stats', $this->hook_stats);
    }
    
    /**
     * Reset enhanced hook statistics
     */
    public function reset_hook_stats() {
        $this->hook_stats = array(
            'posts_processed' => 0,
            'posts_deleted' => 0,
            'comprehensive_processes' => 0,
            'last_activity' => current_time('mysql')
        );
        
        update_option('aiox_enhanced_hook_stats', $this->hook_stats);
    }
    
    /**
     * Schedule comprehensive processing for a post
     */
    public function schedule_comprehensive_processing($post_id, $delay = 10) {
        wp_schedule_single_event(time() + $delay, 'aiox_process_post_comprehensive', array($post_id));
        AIOX_Logger::debug("Enhanced Hooks: Scheduled comprehensive processing for post {$post_id} with {$delay}s delay");
    }
    
    /**
     * Schedule component synchronization for a post
     */
    public function schedule_component_sync($post_id, $action = 'update', $delay = 5) {
        wp_schedule_single_event(time() + $delay, 'aiox_sync_post_components', array($post_id, $action));
        AIOX_Logger::debug("Enhanced Hooks: Scheduled component sync for post {$post_id} (action: {$action}) with {$delay}s delay");
    }
    
    /**
     * Check if comprehensive processing is enabled
     */
    public function is_comprehensive_processing_enabled() {
        return $this->comprehensive_sync->is_comprehensive_processing_enabled();
    }
    
    /**
     * Enable comprehensive processing
     */
    public function enable_comprehensive_processing() {
        $this->comprehensive_sync->enable_comprehensive_processing();
    }
    
    /**
     * Disable comprehensive processing
     */
    public function disable_comprehensive_processing() {
        $this->comprehensive_sync->disable_comprehensive_processing();
    }
}
