<?php
/**
 * AIOX Enhanced Microdata
 * 
 * Implements comprehensive Schema.org microdata including:
 * - JSON-LD structured data
 * - HTML microdata attributes
 * - FAQPage schema for Q&A content
 * - AIOX-specific metadata as AdditionalProperty
 * - Educational level, language, and semantic context
 * 
 * @since 5.2.1
 */

if (!defined('ABSPATH')) {
    exit;
}

class AIOX_Enhanced_Microdata {

    /**
     * Constructor
     */
    public function __construct() {
        // JSON-LD in head
        add_action('wp_head', array($this, 'output_json_ld'), 5);
        
        // HTML microdata attributes on content
        add_filter('the_content', array($this, 'add_microdata_to_content'), 999);
        
        // Body class for microdata
        add_filter('body_class', array($this, 'add_schema_body_class'));
    }

    /**
     * Output enhanced JSON-LD structured data
     */
    public function output_json_ld() {
        if (!is_singular()) {
            return;
        }

        global $post;
        $post_id = $post->ID;

        // Get AIOX data
        $ingredients = get_post_meta($post_id, '_aiox_ingredients', true);
        $metadata = get_post_meta($post_id, '_aiox_metadata', true);
        $qa_pairs = get_post_meta($post_id, '_aiox_qa_pairs', true);
        $processed_data = get_post_meta($post_id, '_aiox_processed_data', true);

        if (empty($ingredients) && empty($qa_pairs)) {
            return;
        }

        // Build Article schema
        $schemas = array();
        $schemas[] = $this->build_article_schema($post, $ingredients, $metadata, $processed_data);

        // Add FAQPage schema if Q&A exists AND feature is enabled
        if (!empty($qa_pairs) && is_array($qa_pairs) && AIOX_Utils::is_feature_enabled('schema_markup')) {
            $schemas[] = $this->build_faq_schema($post, $qa_pairs);
        }

        // Output all schemas
        foreach ($schemas as $schema) {
            echo '<script type="application/ld+json">' . "\n";
            echo wp_json_encode($schema, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
            echo "\n" . '</script>' . "\n";
        }
    }

    /**
     * Build comprehensive Article schema
     */
    private function build_article_schema($post, $ingredients, $metadata, $processed_data) {
        $post_id = $post->ID;
        
        $schema = array(
            '@context' => 'https://schema.org',
            '@type' => 'Article',
            '@id' => get_permalink($post_id) . '#article',
            'headline' => get_the_title($post_id),
            'url' => get_permalink($post_id),
            'datePublished' => get_the_date('c', $post_id),
            'dateModified' => get_the_modified_date('c', $post_id),
            'author' => array(
                '@type' => 'Person',
                'name' => get_the_author_meta('display_name', $post->post_author),
                'url' => get_author_posts_url($post->post_author)
            ),
            'publisher' => array(
                '@type' => 'Organization',
                'name' => get_bloginfo('name'),
                'url' => home_url(),
                'logo' => array(
                    '@type' => 'ImageObject',
                    'url' => get_site_icon_url()
                )
            )
        );

        // Add description
        $excerpt = get_the_excerpt($post_id);
        if ($excerpt) {
            $schema['description'] = wp_strip_all_tags($excerpt);
        }

        // Add keywords from ingredients
        if (is_array($ingredients)) {
            $keywords = array();
            foreach ($ingredients as $ingredient) {
                if (isset($ingredient['metadata']['keywords'])) {
                    $keywords = array_merge($keywords, $ingredient['metadata']['keywords']);
                }
            }
            if (!empty($keywords)) {
                $schema['keywords'] = implode(', ', array_unique($keywords));
            }
        }

        // Add thumbnail image
        if (has_post_thumbnail($post_id)) {
            $thumbnail_id = get_post_thumbnail_id($post_id);
            $thumbnail_url = wp_get_attachment_image_url($thumbnail_id, 'full');
            $thumbnail_meta = wp_get_attachment_metadata($thumbnail_id);
            
            $schema['image'] = array(
                '@type' => 'ImageObject',
                'url' => $thumbnail_url,
                'width' => $thumbnail_meta['width'] ?? 800,
                'height' => $thumbnail_meta['height'] ?? 600
            );
        }

        // Add language
        $schema['inLanguage'] = get_locale();

        // Add educational level from metadata
        if (isset($metadata['target_audience'])) {
            $audience_map = array(
                'novice' => 'Beginner',
                'intermediate' => 'Intermediate',
                'expert' => 'Advanced',
                'general' => 'General'
            );
            
            if (isset($audience_map[$metadata['target_audience']])) {
                $schema['educationalLevel'] = $audience_map[$metadata['target_audience']];
            }
        }

        // Add about (domain classification)
        if (isset($metadata['domain']) && !empty($metadata['domain'])) {
            $schema['about'] = array(
                '@type' => 'Thing',
                'name' => ucfirst($metadata['domain'])
            );
        }

        // Add genre/narrative type
        if (isset($metadata['narrative_type'])) {
            $schema['genre'] = ucfirst($metadata['narrative_type']);
        }

        // Add AIOX-specific metadata as additionalProperty
        $aiox_properties = array();
        
        if (isset($metadata['intent'])) {
            $aiox_properties[] = array(
                '@type' => 'PropertyValue',
                'name' => 'aiox:intent',
                'value' => $metadata['intent']
            );
        }
        
        if (isset($metadata['tone'])) {
            $aiox_properties[] = array(
                '@type' => 'PropertyValue',
                'name' => 'aiox:tone',
                'value' => $metadata['tone']
            );
        }
        
        if (isset($metadata['utility'])) {
            $aiox_properties[] = array(
                '@type' => 'PropertyValue',
                'name' => 'aiox:utility',
                'value' => $metadata['utility']
            );
        }
        
        if (isset($metadata['orientation'])) {
            $aiox_properties[] = array(
                '@type' => 'PropertyValue',
                'name' => 'aiox:orientation',
                'value' => $metadata['orientation']
            );
        }
        
        if (isset($metadata['domain'])) {
            $aiox_properties[] = array(
                '@type' => 'PropertyValue',
                'name' => 'aiox:domain',
                'value' => $metadata['domain']
            );
        }
        
        if (isset($metadata['target_audience'])) {
            $aiox_properties[] = array(
                '@type' => 'PropertyValue',
                'name' => 'aiox:targetAudience',
                'value' => $metadata['target_audience']
            );
        }
        
        if (isset($metadata['narrative_type'])) {
            $aiox_properties[] = array(
                '@type' => 'PropertyValue',
                'name' => 'aiox:narrativeType',
                'value' => $metadata['narrative_type']
            );
        }
        
        if (isset($metadata['media_type'])) {
            $aiox_properties[] = array(
                '@type' => 'PropertyValue',
                'name' => 'aiox:mediaType',
                'value' => $metadata['media_type']
            );
        }
        
        if (isset($metadata['geolocation_sensitivity'])) {
            $aiox_properties[] = array(
                '@type' => 'PropertyValue',
                'name' => 'aiox:geolocationSensitivity',
                'value' => $metadata['geolocation_sensitivity']
            );
        }
        
        if (isset($metadata['market_segments'])) {
            $aiox_properties[] = array(
                '@type' => 'PropertyValue',
                'name' => 'aiox:marketSegments',
                'value' => is_array($metadata['market_segments']) ? implode(', ', $metadata['market_segments']) : $metadata['market_segments']
            );
        }

        // Add AIOX processing info
        $aiox_properties[] = array(
            '@type' => 'PropertyValue',
            'name' => 'aiox:processed',
            'value' => 'true'
        );
        
        $aiox_properties[] = array(
            '@type' => 'PropertyValue',
            'name' => 'aiox:version',
            'value' => defined('AIOX_VERSION') ? AIOX_VERSION : '5.2.0'
        );
        
        if (is_array($ingredients)) {
            $aiox_properties[] = array(
                '@type' => 'PropertyValue',
                'name' => 'aiox:ingredientsCount',
                'value' => count($ingredients)
            );
        }
        
        if (isset($processed_data['processing_method'])) {
            $aiox_properties[] = array(
                '@type' => 'PropertyValue',
                'name' => 'aiox:processingMethod',
                'value' => $processed_data['processing_method']
            );
        }

        if (!empty($aiox_properties)) {
            $schema['additionalProperty'] = $aiox_properties;
        }

        // Add word count
        $word_count = str_word_count(wp_strip_all_tags($post->post_content));
        if ($word_count > 0) {
            $schema['wordCount'] = $word_count;
        }

        return $schema;
    }

    /**
     * Build FAQPage schema from Q&A pairs
     */
    private function build_faq_schema($post, $qa_pairs) {
        $schema = array(
            '@context' => 'https://schema.org',
            '@type' => 'FAQPage',
            '@id' => get_permalink($post->ID) . '#faq',
            'mainEntity' => array()
        );

        foreach ($qa_pairs as $qa) {
            if (empty($qa['question']) || empty($qa['answer'])) {
                continue;
            }

            $schema['mainEntity'][] = array(
                '@type' => 'Question',
                'name' => $qa['question'],
                'acceptedAnswer' => array(
                    '@type' => 'Answer',
                    'text' => $qa['answer']
                )
            );
        }

        return $schema;
    }

    /**
     * Add HTML microdata attributes to content
     */
    public function add_microdata_to_content($content) {
        if (!is_singular() || is_feed()) {
            return $content;
        }

        global $post;
        $post_id = $post->ID;

        // Get AIOX data
        $metadata = get_post_meta($post_id, '_aiox_metadata', true);
        $qa_pairs = get_post_meta($post_id, '_aiox_qa_pairs', true);

        if (empty($metadata) && empty($qa_pairs)) {
            return $content;
        }

        // Wrap content in Article itemscope
        $microdata_attrs = array(
            'itemscope',
            'itemtype="https://schema.org/Article"'
        );

        // Add AIOX metadata as data attributes for AI crawlers
        if (is_array($metadata)) {
            if (isset($metadata['intent'])) {
                $microdata_attrs[] = 'data-aiox-intent="' . esc_attr($metadata['intent']) . '"';
            }
            if (isset($metadata['tone'])) {
                $microdata_attrs[] = 'data-aiox-tone="' . esc_attr($metadata['tone']) . '"';
            }
            if (isset($metadata['domain'])) {
                $microdata_attrs[] = 'data-aiox-domain="' . esc_attr($metadata['domain']) . '"';
            }
            if (isset($metadata['utility'])) {
                $microdata_attrs[] = 'data-aiox-utility="' . esc_attr($metadata['utility']) . '"';
            }
            if (isset($metadata['target_audience'])) {
                $microdata_attrs[] = 'data-aiox-audience="' . esc_attr($metadata['target_audience']) . '"';
            }
            if (isset($metadata['narrative_type'])) {
                $microdata_attrs[] = 'data-aiox-narrative="' . esc_attr($metadata['narrative_type']) . '"';
            }
        }

        // Wrap main content
        $enhanced_content = '<div ' . implode(' ', $microdata_attrs) . '>';
        
        // Add hidden meta tags for schema properties
        $enhanced_content .= '<meta itemprop="headline" content="' . esc_attr(get_the_title($post_id)) . '">';
        $enhanced_content .= '<meta itemprop="datePublished" content="' . esc_attr(get_the_date('c', $post_id)) . '">';
        $enhanced_content .= '<meta itemprop="dateModified" content="' . esc_attr(get_the_modified_date('c', $post_id)) . '">';
        
        // Author microdata
        $enhanced_content .= '<div itemprop="author" itemscope itemtype="https://schema.org/Person" style="display:none;">';
        $enhanced_content .= '<meta itemprop="name" content="' . esc_attr(get_the_author_meta('display_name', $post->post_author)) . '">';
        $enhanced_content .= '</div>';

        // Add main article body
        $enhanced_content .= '<div itemprop="articleBody">';
        $enhanced_content .= $content;
        $enhanced_content .= '</div>';

        $enhanced_content .= '</div>'; // Close article wrapper

        return $enhanced_content;
    }

    /**
     * Add schema body class
     */
    public function add_schema_body_class($classes) {
        if (is_singular()) {
            global $post;
            
            $ingredients = get_post_meta($post->ID, '_aiox_ingredients', true);
            if (!empty($ingredients)) {
                $classes[] = 'aiox-enhanced';
                $classes[] = 'schema-microdata-enabled';
            }
        }
        
        return $classes;
    }

    /**
     * Add FAQ section CSS
     */
    public static function add_faq_styles() {
        if (!is_singular()) {
            return;
        }
        ?>
        <style type="text/css">
        .aiox-faq-section {
            margin: 30px 0;
            padding: 20px;
            background: #f9f9f9;
            border-left: 4px solid #667eea;
            border-radius: 4px;
        }
        
        .aiox-faq-title {
            margin-top: 0;
            color: #333;
            font-size: 1.5em;
            margin-bottom: 20px;
        }
        
        .aiox-faq-item {
            margin-bottom: 20px;
            padding-bottom: 20px;
            border-bottom: 1px solid #e0e0e0;
        }
        
        .aiox-faq-item:last-child {
            border-bottom: none;
            margin-bottom: 0;
            padding-bottom: 0;
        }
        
        .aiox-faq-question {
            margin: 0 0 10px 0;
            color: #667eea;
            font-size: 1.2em;
            font-weight: 600;
        }
        
        .aiox-faq-answer {
            color: #555;
            line-height: 1.6;
        }
        
        .aiox-faq-answer p {
            margin: 0;
        }
        
        /* Make FAQ section print-friendly */
        @media print {
            .aiox-faq-section {
                background: white;
                border-left: 2px solid #000;
            }
        }
        </style>
        <?php
    }
}

// Initialize if enabled
if (get_option('aiox_enhanced_microdata_enabled', true)) {
    new AIOX_Enhanced_Microdata();
    add_action('wp_head', array('AIOX_Enhanced_Microdata', 'add_faq_styles'), 100);
}