<?php
/**
 * AIOX Hooks Class
 * 
 * Handles WordPress hooks and integration
 */

if (!defined('ABSPATH')) {
    exit;
}

class AIOX_Hooks {
    
    /**
     * Constructor
     */
    public function __construct() {
        $this->init_hooks();
    }
    
    /**
     * Initialize WordPress hooks
     */
    private function init_hooks() {
        // WordPress hooks
        add_action('save_post', array($this, 'on_post_save'), 10, 3);
        add_action('delete_post', array($this, 'on_post_delete'));
        add_action('transition_post_status', array($this, 'on_post_status_change'), 10, 3);
        add_action('admin_init', array($this, 'on_admin_init'));
        add_action('wp_head', array($this, 'add_meta_tags'));
        add_action('wp_footer', array($this, 'add_footer_content'));
        
        // Handle .well-known file requests
        add_action('init', array($this, 'add_rewrite_rules'));
        add_filter('query_vars', array($this, 'add_query_vars'));
        add_action('template_redirect', array($this, 'handle_well_known_requests'));
        
        // AJAX hooks
        add_action('wp_ajax_aiox_generate_manifest', array($this, 'ajax_generate_manifest'));
        add_action('wp_ajax_aiox_test_connection', array($this, 'ajax_test_connection'));
        add_action('wp_ajax_aiox_process_post', array($this, 'ajax_process_post'));
        add_action('wp_ajax_aiox_process_queue_now', array($this, 'ajax_process_queue_now'));

        add_action('wp_ajax_aiox_create_ai_info_page', array($this, 'ajax_create_ai_info_page'));
        add_action('wp_ajax_aiox_get_file_status', array($this, 'ajax_get_file_status'));
        
        // Cron hooks
        add_action('aiox_process_queue', array($this, 'process_queue'));
        add_action('aiox_cleanup_logs', array($this, 'cleanup_logs'));
        
        // REST API hooks
        add_action('rest_api_init', array($this, 'register_rest_routes'));
    }
    
    /**
     * Handle post save (Version 2.0.1+: Manual processing only)
     */
    public function on_post_save($post_id, $post) {
        // Version 2.0.1+: Check automatic processing setting
        $automatic_processing = get_option('aiox_enable_automatic_processing', false);
        
        if (!$automatic_processing) {
            // Manual processing only - no automatic processing
            return;
        }
        
        // Automatic processing enabled - trigger AI-only processing
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return;
        }
        
        if (!current_user_can('edit_post', $post_id)) {
            return;
        }
        
        if (wp_is_post_revision($post_id)) {
            return;
        }
        
        // Only process published posts and pages
        if (!in_array($post->post_status, array('publish'))) {
            return;
        }
        
        if (!in_array($post->post_type, array('post', 'page'))) {
            return;
        }
        
        // Trigger AI-only processing
        require_once AIOX_PLUGIN_DIR . 'core/class-aiox-ai-only-processor.php';
        $ai_processor = new AIOX_AI_Only_Processor();
        
        // Check if API keys are configured
        if ($ai_processor->is_api_configured()) {
            // Process in background to avoid slowing down post save
            wp_schedule_single_event(time() + 10, 'aiox_process_post_background', array($post_id));
        }
    }
    
    /**
     * Handle post deletion
     */
    public function on_post_delete($post_id) {
        $post = get_post($post_id);
        
        if (!$post || !AIOX_Utils::should_process_post_type($post->post_type)) {
            return;
        }
        
        // Log removal for delta feed before cleanup
        self::log_content_removal($post_id, 'Content permanently deleted');
        
        // Remove from queue
        $this->remove_from_queue($post_id);
        
        // Clean up generated files
        $this->cleanup_post_files($post_id);
        
        AIOX_Logger::info('Post removed from processing', array(
            'post_id' => $post_id,
            'post_title' => $post->post_title,
            'post_type' => $post->post_type,
            'action' => 'delete'
        ));
    }
    
    /**
     * Handle post status changes
     */
    public function on_post_status_change($new_status, $old_status, $post) {
        if (!AIOX_Utils::should_process_post_type($post->post_type)) {
            return;
        }
        
        if ($new_status === 'publish' && $old_status !== 'publish') {
            // Post was published
            $this->add_to_queue($post->ID, 'publish');
        } elseif ($old_status === 'publish' && $new_status !== 'publish') {
            // Post was unpublished - log removal for delta feed
            $reason = $new_status === 'trash' ? 'Content moved to trash' : 'Content unpublished (status: ' . $new_status . ')';
            self::log_content_removal($post->ID, $reason);
            
            $this->remove_from_queue($post->ID);
            $this->cleanup_post_files($post->ID);
        }
        
        AIOX_Logger::debug('Post status changed', array(
            'post_id' => $post->ID,
            'old_status' => $old_status,
            'new_status' => $new_status
        ));
    }
    
    /**
     * Admin initialization
     */
    public function on_admin_init() {
        // Update last activity
        AIOX_Utils::update_option('last_activity', current_time('mysql'));
        
        // Check for plugin updates or configuration changes
        $this->check_plugin_updates();
    }
    
    /**
     * Show admin notices
     */
    public function show_admin_notices() {
        // Check system requirements
        $requirements = AIOX_Utils::check_system_requirements();
        $failed_requirements = array_filter($requirements, function($req) {
            return !$req['status'];
        });
        
        if (!empty($failed_requirements)) {
            echo '<div class="notice notice-warning">';
            echo '<p><strong>AIOX Publisher Suite Pro:</strong> Some system requirements are not met:</p>';
            echo '<ul>';
            foreach ($failed_requirements as $req) {
                echo '<li>' . esc_html($req['name']) . ': ' . esc_html($req['current']) . ' (Required: ' . esc_html($req['required']) . ')</li>';
            }
            echo '</ul>';
            echo '</div>';
        }
        
        // Show configuration notices
        if (!AIOX_Utils::get_option('license_key') && current_user_can('manage_options')) {
            echo '<div class="notice notice-info is-dismissible">';
            echo '<p><strong>AIOX Publisher Suite Pro:</strong> ';
            echo 'Enter your license key in <a href="' . admin_url('admin.php?page=aiox-settings') . '">Settings</a> to unlock Pro features.';
            echo '</p>';
            echo '</div>';
        }
    }
    
    /**
     * Add meta tags to head
     */
    public function add_meta_tags() {
        if (!is_singular()) {
            return;
        }
        
        // Add AIOX meta tags - using data attributes to avoid search snippet issues
        // These are for AI crawlers only, not for search engine snippets
        echo '<!-- AIOX AI Discovery -->' . "\n";
        echo '<link rel="ai-manifest" href="' . home_url('/aio.json') . '">' . "\n";
        echo '<link rel="alternate" type="application/json" href="' . home_url('/.well-known/aio.json') . '" title="AI Manifest">' . "\n";
        
        // Add OpenGraph and Twitter Cards if enabled (for AI visibility score)
        if (get_option('aiox_enable_social_meta', true)) {
            $this->add_social_meta_tags();
        }
        
        // Add structured data if enabled
        if (AIOX_Utils::is_feature_enabled('schema_markup')) {
            $this->add_structured_data();
        }
    }
    
    /**
     * Add OpenGraph and Twitter Card meta tags for AI visibility
     * Only adds if not already present (respects Yoast, Rank Math, etc.)
     */
    private function add_social_meta_tags() {
        global $post;
        
        if (!$post) {
            return;
        }
        
        // Check if OpenGraph tags already exist (from Yoast, Rank Math, etc.)
        if ($this->has_existing_og_tags()) {
            return;
        }
        
        $post_id = $post->ID;
        $title = get_the_title($post_id);
        $url = get_permalink($post_id);
        $site_name = get_bloginfo('name');
        $image = '';
        $description = '';
        
        // Get AIOX processed data for better description
        $processed_data = get_post_meta($post_id, '_aiox_processed_data', true);
        
        // Priority for description: AIOX summary > excerpt > content excerpt
        if (!empty($processed_data['summary'])) {
            $description = $processed_data['summary'];
        } elseif (has_excerpt($post_id)) {
            $description = get_the_excerpt($post_id);
        } else {
            // Generate from content
            $content = get_post_field('post_content', $post_id);
            $description = wp_trim_words(wp_strip_all_tags($content), 30, '...');
        }
        
        // Get featured image
        if (has_post_thumbnail($post_id)) {
            $image = get_the_post_thumbnail_url($post_id, 'large');
        }
        
        // Try to get stored OpenGraph data from AIOX processing
        $stored_og = get_post_meta($post_id, '_aiox_open_graph', true);
        if (is_array($stored_og)) {
            $title = !empty($stored_og['title']) ? $stored_og['title'] : $title;
            $description = !empty($stored_og['description']) ? $stored_og['description'] : $description;
            $image = !empty($stored_og['image']) ? $stored_og['image'] : $image;
        }
        
        // Fallback image: site logo or site icon
        if (empty($image)) {
            // Try custom logo first
            $custom_logo_id = get_theme_mod('custom_logo');
            if ($custom_logo_id) {
                $image = wp_get_attachment_image_url($custom_logo_id, 'full');
            }
            // Then try site icon
            if (empty($image)) {
                $site_icon_id = get_option('site_icon');
                if ($site_icon_id) {
                    $image = wp_get_attachment_image_url($site_icon_id, 'full');
                }
            }
        }
        
        // Sanitize description - strip tags, normalize whitespace, limit length
        $description = wp_strip_all_tags($description);
        $description = preg_replace('/\s+/', ' ', trim($description));
        $description = mb_substr($description, 0, 200);
        
        echo "\n<!-- AIOX Social Meta Tags -->\n";
        
        // OpenGraph tags
        echo '<meta property="og:type" content="article">' . "\n";
        echo '<meta property="og:title" content="' . esc_attr($title) . '">' . "\n";
        echo '<meta property="og:description" content="' . esc_attr($description) . '">' . "\n";
        echo '<meta property="og:url" content="' . esc_url($url) . '">' . "\n";
        echo '<meta property="og:site_name" content="' . esc_attr($site_name) . '">' . "\n";
        
        if ($image) {
            echo '<meta property="og:image" content="' . esc_url($image) . '">' . "\n";
        }
        
        // Twitter Card tags
        echo '<meta name="twitter:card" content="summary_large_image">' . "\n";
        echo '<meta name="twitter:title" content="' . esc_attr($title) . '">' . "\n";
        echo '<meta name="twitter:description" content="' . esc_attr($description) . '">' . "\n";
        
        if ($image) {
            echo '<meta name="twitter:image" content="' . esc_url($image) . '">' . "\n";
        }
        
        echo "<!-- /AIOX Social Meta Tags -->\n";
    }
    
    /**
     * Check if OpenGraph tags already exist in the page
     * (from plugins like Yoast SEO, Rank Math, All in One SEO, etc.)
     */
    private function has_existing_og_tags() {
        // Check common plugin indicators
        if (defined('WPSEO_VERSION')) {
            return true; // Yoast SEO
        }
        if (class_exists('RankMath')) {
            return true; // Rank Math
        }
        if (class_exists('All_in_One_SEO_Pack')) {
            return true; // All in One SEO
        }
        if (class_exists('The_SEO_Framework')) {
            return true; // The SEO Framework
        }
        if (defined('JEPACK__VERSION') && class_exists('Jetpack')) {
            // Check if Jetpack Open Graph is enabled
            if (class_exists('Jetpack') && Jetpack::is_module_active('publicize')) {
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Add footer content
     */
    public function add_footer_content() {
        if (!is_singular()) {
            return;
        }
        
        // Add AI badge if enabled
        if (AIOX_Utils::is_feature_enabled('ai_badge')) {
            $this->add_ai_badge();
        }
    }
    
    /**
     * AJAX: Generate manifest
     */
    public function ajax_generate_manifest() {
        check_ajax_referer('aiox_ajax_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized access');
            return;
        }
        
        try {
            // Generate all AIOX files
            $result = $this->generate_all_files();
            
            if ($result) {
                // Clear the regeneration notice flag
                delete_option('aiox_needs_file_regeneration');
                
                AIOX_Logger::info('All AIOX files regenerated via AJAX', AIOX_Utils::get_current_user_info());
                wp_send_json_success('All AIOX files regenerated successfully (including capsule files)');
            } else {
                wp_send_json_error('File generation returned false');
            }
            
        } catch (Exception $e) {
            AIOX_Logger::error('Failed to generate manifest via AJAX', array(
                'error' => $e->getMessage(),
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'user' => AIOX_Utils::get_current_user_info()
            ));
            
            wp_send_json_error('Failed to regenerate files: ' . $e->getMessage());
        } catch (Error $e) {
            // Catch PHP 7+ fatal errors
            AIOX_Logger::error('PHP Error during manifest generation', array(
                'error' => $e->getMessage(),
                'file' => $e->getFile(),
                'line' => $e->getLine()
            ));
            
            wp_send_json_error('PHP Error: ' . $e->getMessage());
        }
    }
    
    /**
     * AJAX: Test AI connection
     */
    public function ajax_test_connection() {
        check_ajax_referer('aiox_ajax_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }
        
        $endpoint = AIOX_Utils::get_ai_endpoint();
        $license_key = AIOX_Utils::get_option('license_key');
        
        if (empty($license_key)) {
            wp_send_json_error('License key is required for AI connection');
            return;
        }
        
        $response = wp_remote_post($endpoint, array(
            'timeout' => 10,
            'headers' => array(
                'Authorization' => 'Bearer ' . $license_key,
                'Content-Type' => 'application/json'
            ),
            'body' => json_encode(array('action' => 'test'))
        ));
        
        if (is_wp_error($response)) {
            wp_send_json_error('Connection failed: ' . $response->get_error_message());
            return;
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        
        if ($response_code === 200) {
            wp_send_json_success('Connection successful');
        } else {
            wp_send_json_error('Connection failed with status: ' . $response_code);
        }
    }
    
    /**
     * AJAX: Process a specific post
     */
    public function ajax_process_post() {
        check_ajax_referer('aiox_ajax_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }
        
        $post_id = intval($_POST['post_id'] ?? 0);
        
        if (!$post_id) {
            wp_send_json_error('Invalid post ID');
            return;
        }
        
        $generator = new AIOX_Generator();
        $success = $generator->process_post($post_id);
        
        if ($success) {
            $qa_count = get_post_meta($post_id, '_aiox_qa_count', true);
            wp_send_json_success(array(
                'message' => 'Post processed successfully',
                'qa_count' => $qa_count
            ));
        } else {
            wp_send_json_error('Failed to process post');
        }
    }
    
    /**
     * AJAX: Process queue immediately
     */
    public function ajax_process_queue_now() {
        check_ajax_referer('aiox_ajax_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }
        
        $this->process_queue();
        
        wp_send_json_success('Queue processed');
    }
    
    /**
     * AJAX: Create AI info page
     */
    public function ajax_create_ai_info_page() {
        check_ajax_referer('aiox_ajax_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }
        
        $generator = new AIOX_Generator();
        $post_id = $generator->create_ai_info_page(true); // Force creation
        
        if ($post_id) {
            $post_url = get_permalink($post_id);
            wp_send_json_success('AI info page created successfully! <a href="' . esc_url($post_url) . '" target="_blank">View page</a>');
        } else {
            wp_send_json_error('Failed to create AI info page');
        }
    }
    
    /**
     * Generate all AIOX files (manifest, TDM, keys, etc.)
     */
    public function generate_all_files() {
        try {
            $well_known_dir = ABSPATH . '.well-known';
            
            // Ensure .well-known directory exists
            if (!file_exists($well_known_dir)) {
                if (!wp_mkdir_p($well_known_dir)) {
                    throw new Exception('Failed to create .well-known directory');
                }
            }
            
            // Generate manifest file
            $this->generate_manifest();
            
            // Generate TDM-REP file
            $this->generate_tdmrep();
            
            // Generate public key file
            $this->generate_public_key();
            
            // Generate delta feed file
            $this->generate_delta_feed();
            
            // Generate AIOX pack file
            $this->generate_pack_file();
            
            // Generate ingredients file
            if (class_exists('AIOX_Ingredients')) {
                AIOX_Ingredients::export_ingredients_file();
            }
            
            // Generate proof file
            if (class_exists('AIOX_Crypto')) {
                $crypto = new AIOX_Crypto();
                $crypto->generate_proof_file();
            }
            
            // Regenerate capsule files from database (important for reinstall scenario)
            if (class_exists('AIOX_Capsule_Manager')) {
                try {
                    $capsule_manager = new AIOX_Capsule_Manager();
                    $capsule_stats = $capsule_manager->regenerate_all_capsule_files();
                    AIOX_Logger::info('Capsule files regeneration', $capsule_stats);
                } catch (Exception $e) {
                    AIOX_Logger::warning('Capsule regeneration failed', array('error' => $e->getMessage()));
                }
            }
            
            AIOX_Logger::info('All AIOX files generated/updated (including ingredients, proof, and capsule files)');
            
            return true; // Return true on success
            
        } catch (Exception $e) {
            AIOX_Logger::error('Failed to generate AIOX files', array('error' => $e->getMessage()));
            throw $e; // Re-throw for AJAX error handling
        }
    }
    
    /**
     * Generate main AIOX manifest
     * Lightweight version with content index
     */
    public function generate_manifest() {
        $well_known_dir = ABSPATH . '.well-known';
        $manifest_file = $well_known_dir . '/aio.json';
        
        // Ensure directory exists
        if (!file_exists($well_known_dir)) {
            if (!wp_mkdir_p($well_known_dir)) {
                AIOX_Logger::error('Failed to create .well-known directory');
                throw new Exception('Failed to create .well-known directory');
            }
        }
        
        $home_url = home_url();
        $site_name = get_bloginfo('name');
        $options = AIOX_Utils::get_all_options();
        
        // Get inline public key for manifest
        $public_key_inline = null;
        $key_algorithm = 'unknown';
        if (class_exists('AIOX_Crypto')) {
            try {
                $crypto = new AIOX_Crypto();
                $keypair_info = $crypto->get_keypair_info();
                if ($keypair_info && isset($keypair_info['public_key'])) {
                    $public_key_inline = $keypair_info['public_key'];
                    $key_algorithm = $keypair_info['algorithm'] ?? 'ed25519';
                }
            } catch (Exception $e) {
                AIOX_Logger::warning('Could not get crypto keypair', array('error' => $e->getMessage()));
            }
        }
        
        // Get monetization settings
        $monetization = self::get_monetization_settings();
        
        // Get license and enforcement presets
        $license_config = self::get_license_preset_config();
        $monetization_config = self::get_monetization_preset_config();
        $enforcement_config = self::get_enforcement_mode_config();
        
        $manifest = array(
            'aiox_version' => '1.0',
            'manifest_type' => 'lightweight_index',
            'publisher' => array(
                'name' => $site_name,
                'url' => $home_url,
                'attribution_name' => $options['required_attribution'] ?? $site_name,
                'contact' => get_option('aiox_license_contact_email', get_option('admin_email')),
                'public_key' => $public_key_inline ? $key_algorithm . ':' . $public_key_inline : null
            ),
            'licensing' => array(
                'preset' => $license_config['preset'],
                'preset_name' => $license_config['name'],
                'license_code' => $license_config['license_code'],
                'permissions' => array(
                    'training_allowed' => $license_config['training_allowed'],
                    'citation_required' => $license_config['citation_required'],
                    'attribution_required' => $license_config['attribution_required'],
                    'commercial_use' => $license_config['commercial_use'],
                    'derivatives_allowed' => $license_config['derivatives_allowed']
                ),
                'attribution' => $options['required_attribution'] ?? $site_name,
                'terms_url' => get_option('aiox_license_terms_url', $home_url . '/ai-licensing-terms'),
                'custom_terms' => get_option('aiox_license_custom_terms', '')
            ),
            'monetization' => array(
                'preset' => $monetization_config['preset'],
                'preset_name' => $monetization_config['name'],
                'model' => $monetization_config['pricing_model'],
                'subscription_required' => $monetization_config['subscription_required'],
                'pay_per_use' => $monetization_config['pay_per_use'],
                'api_access' => $monetization_config['api_access'],
                'rate_limited' => $monetization_config['rate_limited'],
                'license_api' => $monetization['license_api'],
                'contact' => get_option('aiox_license_contact_email', get_option('admin_email'))
            ),
            'enforcement' => array(
                'mode' => $enforcement_config['mode'],
                'mode_name' => $enforcement_config['name'],
                'compliance_level' => $enforcement_config['compliance_level'],
                'violation_response' => $enforcement_config['violation_response'],
                'legal_binding' => $enforcement_config['legal_binding'],
                'grace_period' => $enforcement_config['grace_period'],
                'message' => $enforcement_config['description']
            ),
            'terms' => array(
                'version' => 'v1',
                'urls' => array(
                    'terms_of_service_url' => get_option('aiox_license_terms_url', $home_url . '/terms-of-service'),
                    'privacy_policy_url' => $home_url . '/privacy-policy'
                ),
                'allow_citation' => $options['allow_citation'] ?? true,
                'allow_training' => $options['allow_training'] ?? false
            ),
            'endpoints' => array(
                'manifest_url' => $home_url . '/.well-known/aio.json',
                'delta_url' => $home_url . '/.well-known/aiox.ndjson',
                'tdmrep_url' => $home_url . '/.well-known/tdmrep.json',
                'public_key_url' => $home_url . '/.well-known/aiox-pubkey.pem',
                'posts_base_url' => $home_url . '/.well-known/posts/',
                'post_detail_template' => $home_url . '/.well-known/posts/{post_id}.json'
            ),
            'features' => array(
                'content_analysis' => true,
                'qa_extraction' => true,
                'cryptographic_signing' => true,
                'delta_feed' => true,
                'individual_post_files' => true,
                'lightweight_index' => true,
                'preset_licensing' => true
            ),
            'generated_at' => current_time('c'),
            'generator' => 'AIOX Publisher Suite Pro v1.0'
        );
        
        // Add pack endpoints if enabled
        if ($options['pack_enabled'] ?? true) {
            $manifest['endpoints']['pack_index_url'] = $home_url . '/.well-known/aiox-pack-index.json';
            $manifest['endpoints']['pack_template'] = $home_url . '/.well-known/packs/aiox-pack-{number}.aiox';
            $manifest['endpoints']['pack_url'] = $home_url . '/.well-known/aiox.aiox'; // Legacy: first pack
            $manifest['endpoints']['pack_media_type'] = 'application/vnd.aiox.pack+json';
            $manifest['features']['paginated_packs'] = true;
        }
        
        // Add lightweight content index instead of full content
        try {
            $manifest['content_index'] = self::get_content_data_with_working_ingredients();
        } catch (Exception $e) {
            AIOX_Logger::warning('Could not get content index', array('error' => $e->getMessage()));
            $manifest['content_index'] = array();
        }
        
        // Add metadata statistics
        if (class_exists('AIOX_Content_Data') && method_exists('AIOX_Content_Data', 'get_content_statistics')) {
            try {
                $manifest['metadata'] = AIOX_Content_Data::get_content_statistics();
            } catch (Exception $e) {
                AIOX_Logger::warning('Could not get content statistics', array('error' => $e->getMessage()));
                $manifest['metadata'] = array();
            }
        } else {
            $manifest['metadata'] = array();
        }
        
        $json_content = wp_json_encode($manifest, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
        
        if ($json_content === false) {
            throw new Exception('Failed to encode manifest as JSON');
        }
        
        $bytes_written = @file_put_contents($manifest_file, $json_content);
        
        if ($bytes_written === false) {
            throw new Exception('Failed to write manifest file to .well-known directory');
        }
        
        // Also create aio.json at site root for AI crawler compatibility
        // Many AI tools look for /aio.json at root, not just /.well-known/aio.json
        $root_manifest_file = ABSPATH . 'aio.json';
        @file_put_contents($root_manifest_file, $json_content);
        
        AIOX_Logger::info('AIOX manifest generated (lightweight version)', array(
            'file' => $manifest_file,
            'content_items' => count($manifest['content_index']),
            'file_size' => $bytes_written,
            'file_size_human' => size_format($bytes_written)
        ));
    }

    /**
     * Get monetization settings for manifest
     * Returns structured monetization options per AIOX protocol
     */
    private static function get_monetization_settings() {
        $home_url = home_url();
        
        // Get monetization options from settings
        $subscription_required = get_option('aiox_monetization_subscription', false);
        $pay_per_use_enabled = get_option('aiox_monetization_pay_per_use', false);
        $license_api_url = get_option('aiox_monetization_license_api', '');
        $pricing_tier = get_option('aiox_monetization_pricing_tier', 'free');
        
        $monetization = array(
            'model' => $pricing_tier, // free, freemium, subscription, pay_per_use
            'subscription_required' => (bool) $subscription_required,
            'free_access' => !$subscription_required && !$pay_per_use_enabled
        );
        
        // Add pay-per-use API if enabled
        if ($pay_per_use_enabled && !empty($license_api_url)) {
            $monetization['pay_per_use'] = array(
                'enabled' => true,
                'api_endpoint' => $license_api_url
            );
        }
        
        // Add licensing API endpoint if configured
        if (!empty($license_api_url)) {
            $monetization['license_api'] = $license_api_url;
        } else {
            // Default to a standard endpoint structure
            $monetization['license_api'] = $home_url . '/wp-json/aiox/v1/license';
        }
        
        // Add contact for licensing inquiries
        $monetization['contact'] = get_option('admin_email');
        
        return $monetization;
    }
    
    /**
     * Get license preset configuration
     * Returns the active license preset with all its settings
     */
    private static function get_license_preset_config() {
        $preset = get_option('aiox_license_preset', 'attribution_required');
        
        $presets = array(
            'open_access' => array(
                'preset' => 'open_access',
                'name' => 'Open Access',
                'training_allowed' => true,
                'citation_required' => true,
                'attribution_required' => true,
                'commercial_use' => true,
                'derivatives_allowed' => true,
                'share_alike' => false,
                'license_code' => 'AIOX-OA-1.0',
                'comparable_to' => 'CC BY 4.0'
            ),
            'attribution_required' => array(
                'preset' => 'attribution_required',
                'name' => 'Attribution Required',
                'training_allowed' => false,
                'citation_required' => true,
                'attribution_required' => true,
                'commercial_use' => true,
                'derivatives_allowed' => true,
                'share_alike' => true,
                'license_code' => 'AIOX-AR-1.0',
                'comparable_to' => 'CC BY-SA 4.0'
            ),
            'commercial_license' => array(
                'preset' => 'commercial_license',
                'name' => 'Commercial License',
                'training_allowed' => false,
                'citation_required' => true,
                'attribution_required' => true,
                'commercial_use' => false,
                'derivatives_allowed' => false,
                'share_alike' => false,
                'license_code' => 'AIOX-CL-1.0',
                'comparable_to' => 'All Rights Reserved'
            )
        );
        
        return $presets[$preset] ?? $presets['attribution_required'];
    }
    
    /**
     * Get monetization preset configuration
     * Returns the active monetization preset with all its settings
     */
    private static function get_monetization_preset_config() {
        $preset = get_option('aiox_monetization_preset', 'free');
        
        $presets = array(
            'free' => array(
                'preset' => 'free',
                'name' => 'Free Access',
                'subscription_required' => false,
                'pay_per_use' => false,
                'api_access' => true,
                'rate_limited' => false,
                'pricing_model' => 'free'
            ),
            'freemium' => array(
                'preset' => 'freemium',
                'name' => 'Freemium',
                'subscription_required' => false,
                'pay_per_use' => true,
                'api_access' => true,
                'rate_limited' => true,
                'pricing_model' => 'freemium'
            ),
            'paid' => array(
                'preset' => 'paid',
                'name' => 'Paid Access',
                'subscription_required' => true,
                'pay_per_use' => true,
                'api_access' => true,
                'rate_limited' => true,
                'pricing_model' => 'paid'
            )
        );
        
        return $presets[$preset] ?? $presets['free'];
    }
    
    /**
     * Get enforcement mode configuration
     * Returns the active enforcement mode with all its settings
     */
    private static function get_enforcement_mode_config() {
        $mode = get_option('aiox_enforcement_mode', 'flexible');
        
        $modes = array(
            'strict' => array(
                'mode' => 'strict',
                'name' => 'Strict Enforcement',
                'description' => 'These are requirements, not suggestions. AI systems that violate these terms are considered non-compliant.',
                'compliance_level' => 'mandatory',
                'violation_response' => 'block',
                'legal_binding' => true,
                'grace_period' => false
            ),
            'flexible' => array(
                'mode' => 'flexible',
                'name' => 'Flexible Enforcement',
                'description' => 'These are strong preferences. AI systems may proceed in good faith while the AIOX protocol gains adoption.',
                'compliance_level' => 'recommended',
                'violation_response' => 'warn',
                'legal_binding' => false,
                'grace_period' => true
            )
        );
        
        return $modes[$mode] ?? $modes['flexible'];
    }
    
    /**
     * Generate TDM-REP file
     */
    public function generate_tdmrep() {
        $well_known_dir = ABSPATH . '.well-known';
        $tdmrep_file = $well_known_dir . '/tdmrep.json';
        
        $options = AIOX_Utils::get_all_options();
        $home_url = home_url();
        
        // Get preset configurations
        $license_config = self::get_license_preset_config();
        $monetization_config = self::get_monetization_preset_config();
        $enforcement_config = self::get_enforcement_mode_config();
        
        $tdmrep = array(
            'version' => '1.0',
            'licensing' => array(
                'preset' => $license_config['preset'],
                'preset_name' => $license_config['name'],
                'license_code' => $license_config['license_code'],
                'training' => $license_config['training_allowed'],
                'citation_required' => $license_config['citation_required'],
                'attribution_required' => $license_config['attribution_required'],
                'commercial_use' => $license_config['commercial_use'],
                'derivatives_allowed' => $license_config['derivatives_allowed']
            ),
            'policy' => array(
                'default' => $license_config['training_allowed'] ? 'allow' : 'deny',
                'training' => $license_config['training_allowed'],
                'citation' => $license_config['citation_required']
            ),
            'enforcement' => array(
                'mode' => $enforcement_config['mode'],
                'mode_name' => $enforcement_config['name'],
                'compliance_level' => $enforcement_config['compliance_level'],
                'violation_response' => $enforcement_config['violation_response'],
                'legal_binding' => $enforcement_config['legal_binding'],
                'grace_period' => $enforcement_config['grace_period'],
                'message' => $enforcement_config['description']
            ),
            'rules' => array(
                array(
                    'path' => '/',
                    'permission' => $license_config['training_allowed'] ? 'allow' : 'deny',
                    'purpose' => array('training', 'analysis', 'retrieval'),
                    'enforcement' => $enforcement_config['mode']
                )
            ),
            'contact' => array(
                'url' => get_option('aiox_license_terms_url', $home_url . '/contact'),
                'email' => get_option('aiox_license_contact_email', get_option('admin_email')),
                'licensing_inquiries' => get_option('aiox_license_contact_email', get_option('admin_email'))
            ),
            'attribution' => array(
                'required' => $license_config['attribution_required'],
                'name' => $options['required_attribution'] ?? get_bloginfo('name'),
                'url' => $home_url
            ),
            'monetization' => array(
                'preset' => $monetization_config['preset'],
                'preset_name' => $monetization_config['name'],
                'model' => $monetization_config['pricing_model'],
                'subscription_required' => $monetization_config['subscription_required'],
                'pay_per_use' => $monetization_config['pay_per_use'],
                'license_api' => get_option('aiox_monetization_license_api', $home_url . '/wp-json/aiox/v1/license')
            ),
            'custom_terms' => get_option('aiox_license_custom_terms', ''),
            'terms_url' => get_option('aiox_license_terms_url', ''),
            'generated_at' => current_time('c')
        );
        
        $json_content = wp_json_encode($tdmrep, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
        file_put_contents($tdmrep_file, $json_content);
        
        AIOX_Logger::info('TDM-REP file generated with presets', array(
            'file' => $tdmrep_file,
            'license_preset' => $license_config['preset'],
            'monetization_preset' => $monetization_config['preset'],
            'enforcement_mode' => $enforcement_config['mode']
        ));
    }
    
    /**
     * Get all content data from processed posts
     */
    private function get_all_content_data() {
        $content_data = array();
        
        // Get all published posts and pages that have been processed
        $args = array(
            'post_type' => array('post', 'page'),
            'post_status' => 'publish',
            'posts_per_page' => -1,
            'meta_query' => array(
                array(
                    'key' => '_aiox_qa_pairs',
                    'compare' => 'EXISTS'
                )
            )
        );
        
        $posts = get_posts($args);
        
        foreach ($posts as $post) {
            $qa_pairs = get_post_meta($post->ID, '_aiox_qa_pairs', true);
            $processed_at = get_post_meta($post->ID, '_aiox_processed_at', true);
            $qa_count = get_post_meta($post->ID, '_aiox_qa_count', true);
            $method = get_post_meta($post->ID, '_aiox_method', true);
            
            if (!empty($qa_pairs) && is_array($qa_pairs)) {
                $content_item = array(
                    'id' => 'post_' . $post->ID,
                    'title' => $post->post_title,
                    'url' => get_permalink($post->ID),
                    'published' => get_the_date('c', $post->ID),
                    'modified' => get_the_modified_date('c', $post->ID),
                    'author' => get_the_author_meta('display_name', $post->post_author),
                    'excerpt' => get_the_excerpt($post->ID),
                    'post_type' => $post->post_type,
                    'processing' => array(
                        'method' => $method ?: 'basic',
                        'processed_at' => $processed_at,
                        'qa_count' => intval($qa_count)
                    ),
                    'qa_pairs' => array()
                );
                
                // Add Q&A pairs
                foreach ($qa_pairs as $index => $qa) {
                    if (isset($qa['question']) && isset($qa['answer'])) {
                        $content_item['qa_pairs'][] = array(
                            'id' => $index + 1,
                            'question' => $qa['question'],
                            'answer' => $qa['answer'],
                            'confidence' => $qa['confidence'] ?? 0.5,
                            'source' => $qa['source'] ?? 'unknown'
                        );
                    }
                }
                
                // Add metadata
                $content_item['metadata'] = array(
                    'categories' => wp_get_post_categories($post->ID, array('fields' => 'names')),
                    'tags' => wp_get_post_tags($post->ID, array('fields' => 'names')),
                    'word_count' => str_word_count(strip_tags($post->post_content)),
                    'language' => get_locale()
                );
                
                $content_data[] = $content_item;
            }
        }
        
        return $content_data;
    }
    
    /**
     * Get total number of processed posts
     */
    private function get_total_processed_posts() {
        $args = array(
            'post_type' => array('post', 'page'),
            'post_status' => 'publish',
            'posts_per_page' => -1,
            'meta_query' => array(
                array(
                    'key' => '_aiox_qa_pairs',
                    'compare' => 'EXISTS'
                )
            ),
            'fields' => 'ids'
        );
        
        $posts = get_posts($args);
        return count($posts);
    }
    
    /**
     * Get total number of Q&A pairs across all posts
     */
    private function get_total_qa_pairs() {
        $total = 0;
        
        $args = array(
            'post_type' => array('post', 'page'),
            'post_status' => 'publish',
            'posts_per_page' => -1,
            'meta_query' => array(
                array(
                    'key' => '_aiox_qa_count',
                    'compare' => 'EXISTS'
                )
            )
        );
        
        $posts = get_posts($args);
        
        foreach ($posts as $post) {
            $qa_count = get_post_meta($post->ID, '_aiox_qa_count', true);
            $total += intval($qa_count);
        }
        
        return $total;
    }
    
    /**
     * Get last content update timestamp
     */
    private function get_last_content_update() {
        $args = array(
            'post_type' => array('post', 'page'),
            'post_status' => 'publish',
            'posts_per_page' => 1,
            'orderby' => 'modified',
            'order' => 'DESC',
            'meta_query' => array(
                array(
                    'key' => '_aiox_processed_at',
                    'compare' => 'EXISTS'
                )
            )
        );
        
        $posts = get_posts($args);
        
        if (!empty($posts)) {
            return get_the_modified_date('c', $posts[0]->ID);
        }
        
        return current_time('c');
    }
    
    /**
     * Generate public key file
     */
    public function generate_public_key() {
        $well_known_dir = ABSPATH . '.well-known';
        $pubkey_file = $well_known_dir . '/aiox-pubkey.pem';
        
        try {
            // Generate Ed25519 keypair if not exists (only if crypto class exists)
            if (class_exists('AIOX_Crypto')) {
                $crypto = new AIOX_Crypto();
                $keypair = $crypto->get_or_create_keypair();
                
                if ($keypair && isset($keypair['public'])) {
                    $pem_content = "-----BEGIN PUBLIC KEY-----\n";
                    $pem_content .= chunk_split(base64_encode($keypair['public']), 64, "\n");
                    $pem_content .= "-----END PUBLIC KEY-----\n";
                    
                    file_put_contents($pubkey_file, $pem_content);
                    AIOX_Logger::info('Public key file generated', array('file' => $pubkey_file));
                }
            } else {
                // Create placeholder public key file if crypto class not available
                $placeholder_content = "-----BEGIN PUBLIC KEY-----\n";
                $placeholder_content .= "# Placeholder public key - crypto functionality not available\n";
                $placeholder_content .= "# This will be generated when AIOX_Crypto class is loaded\n";
                $placeholder_content .= "-----END PUBLIC KEY-----\n";
                
                file_put_contents($pubkey_file, $placeholder_content);
                AIOX_Logger::info('Placeholder public key file created', array('file' => $pubkey_file));
            }
        } catch (Exception $e) {
            AIOX_Logger::error('Failed to generate public key file', array('error' => $e->getMessage()));
        }
    }
    
    /**
     * Generate delta feed file with proper action types per AIOX protocol
     * Supports: add, update, remove actions with hash tracking
     */
    public function generate_delta_feed() {
        $well_known_dir = ABSPATH . '.well-known';
        $delta_file = $well_known_dir . '/aiox.ndjson';
        $delta_state_file = $well_known_dir . '/aiox-delta-state.json';
        
        try {
            $content = '';
            $home_url = home_url();
            
            // Load previous state to detect changes
            $previous_state = array();
            if (file_exists($delta_state_file)) {
                $previous_state = json_decode(file_get_contents($delta_state_file), true) ?: array();
            }
            
            // Add header entry
            $header_entry = array(
                'feed_version' => '1.0',
                'publisher' => get_bloginfo('name'),
                'publisher_url' => $home_url,
                'last_updated' => current_time('c'),
                'manifest_url' => $home_url . '/.well-known/aio.json'
            );
            $content .= json_encode($header_entry, JSON_UNESCAPED_SLASHES) . "\n";
            
            // Get current processed content
            $current_content = self::get_content_data_with_working_ingredients();
            $current_state = array();
            $changes = array();
            
            foreach ($current_content as $item) {
                $post_id = $item['id'];
                $capsule_id = 'caps-post-' . $post_id;
                
                // Generate content hash for change detection
                $content_hash = self::generate_content_hash($post_id);
                $current_state[$post_id] = array(
                    'hash' => $content_hash,
                    'modified' => $item['modified_at'],
                    'title' => $item['title']
                );
                
                // Determine action type
                if (!isset($previous_state[$post_id])) {
                    // New content - ADD action
                    $changes[] = array(
                        'timestamp' => current_time('c'),
                        'action' => 'add',
                        'capsule_id' => $capsule_id,
                        'post_id' => $post_id,
                        'title' => $item['title'],
                        'url' => $item['url'],
                        'detail_url' => $home_url . '/.well-known/posts/' . $post_id . '.json',
                        'hash' => $content_hash,
                        'ingredients_count' => isset($item['ingredients']['total_count']) ? $item['ingredients']['total_count'] : 0
                    );
                } elseif ($previous_state[$post_id]['hash'] !== $content_hash) {
                    // Content changed - UPDATE action
                    $changes[] = array(
                        'timestamp' => current_time('c'),
                        'action' => 'update',
                        'capsule_id' => $capsule_id,
                        'post_id' => $post_id,
                        'title' => $item['title'],
                        'url' => $item['url'],
                        'detail_url' => $home_url . '/.well-known/posts/' . $post_id . '.json',
                        'hash' => $content_hash,
                        'previous_hash' => $previous_state[$post_id]['hash'],
                        'ingredients_count' => isset($item['ingredients']['total_count']) ? $item['ingredients']['total_count'] : 0
                    );
                }
                // If hash matches, no change needed - skip
            }
            
            // Detect removed content
            $removed_posts = self::get_removed_content_log();
            foreach ($removed_posts as $removed) {
                $changes[] = array(
                    'timestamp' => $removed['timestamp'] ?? current_time('c'),
                    'action' => 'remove',
                    'capsule_id' => 'caps-post-' . $removed['post_id'],
                    'post_id' => $removed['post_id'],
                    'title' => $removed['title'] ?? 'Unknown',
                    'reason' => $removed['reason'] ?? 'Content removed or unpublished',
                    'previous_hash' => $removed['hash'] ?? null
                );
            }
            
            // Also check for posts that existed in previous state but not in current
            foreach ($previous_state as $post_id => $prev_data) {
                if (!isset($current_state[$post_id])) {
                    // Check if already in removed_posts to avoid duplicates
                    $already_logged = false;
                    foreach ($removed_posts as $removed) {
                        if ($removed['post_id'] == $post_id) {
                            $already_logged = true;
                            break;
                        }
                    }
                    
                    if (!$already_logged) {
                        $changes[] = array(
                            'timestamp' => current_time('c'),
                            'action' => 'remove',
                            'capsule_id' => 'caps-post-' . $post_id,
                            'post_id' => $post_id,
                            'title' => $prev_data['title'] ?? 'Unknown',
                            'reason' => 'Content no longer published or processed',
                            'previous_hash' => $prev_data['hash']
                        );
                    }
                }
            }
            
            // Add all changes to delta feed
            foreach ($changes as $change) {
                $content .= json_encode($change, JSON_UNESCAPED_SLASHES) . "\n";
            }
            
            // Write delta feed
            file_put_contents($delta_file, $content);
            
            // Save current state for next comparison
            file_put_contents($delta_state_file, json_encode($current_state, JSON_PRETTY_PRINT));
            
            // Clear the removed content log after processing
            self::clear_removed_content_log();
            
            AIOX_Logger::info('Delta feed file generated (protocol compliant)', array(
                'file' => $delta_file,
                'total_entries' => count($changes) + 1,
                'adds' => count(array_filter($changes, function($c) { return $c['action'] === 'add'; })),
                'updates' => count(array_filter($changes, function($c) { return $c['action'] === 'update'; })),
                'removes' => count(array_filter($changes, function($c) { return $c['action'] === 'remove'; }))
            ));
            
        } catch (Exception $e) {
            AIOX_Logger::error('Failed to generate delta feed file', array('error' => $e->getMessage()));
        }
    }
    
    /**
     * Generate content hash for a post (for delta feed change detection)
     */
    private static function generate_content_hash($post_id) {
        $post = get_post($post_id);
        if (!$post) {
            return null;
        }
        
        // Include content, title, and modification date in hash
        $hash_content = array(
            'title' => $post->post_title,
            'content' => $post->post_content,
            'modified' => $post->post_modified,
            'ingredients_updated' => get_post_meta($post_id, '_aiox_ingredients_updated', true)
        );
        
        return hash('sha256', json_encode($hash_content));
    }
    
    /**
     * Log content removal for delta feed tracking
     * Called when posts are deleted or unpublished
     */
    public static function log_content_removal($post_id, $reason = 'Content removed') {
        $log_file = ABSPATH . '.well-known/aiox-removed-log.json';
        
        $post = get_post($post_id);
        $title = $post ? $post->post_title : 'Unknown';
        $hash = self::generate_content_hash($post_id);
        
        $removed_log = array();
        if (file_exists($log_file)) {
            $removed_log = json_decode(file_get_contents($log_file), true) ?: array();
        }
        
        $removed_log[] = array(
            'post_id' => $post_id,
            'title' => $title,
            'hash' => $hash,
            'reason' => $reason,
            'timestamp' => current_time('c')
        );
        
        file_put_contents($log_file, json_encode($removed_log, JSON_PRETTY_PRINT));
    }
    
    /**
     * Get removed content log
     */
    private static function get_removed_content_log() {
        $log_file = ABSPATH . '.well-known/aiox-removed-log.json';
        
        if (file_exists($log_file)) {
            return json_decode(file_get_contents($log_file), true) ?: array();
        }
        
        return array();
    }
    
    /**
     * Clear removed content log after processing
     */
    private static function clear_removed_content_log() {
        $log_file = ABSPATH . '.well-known/aiox-removed-log.json';
        
        if (file_exists($log_file)) {
            file_put_contents($log_file, '[]');
        }
    }
    
    /**
     * Generate AIOX pack file
     */
    public function generate_pack_file() {
        try {
            if (class_exists('AIOX_Pack')) {
                $pack = new AIOX_Pack();
                $pack->generate_pack();
            } else {
                AIOX_Logger::error('AIOX_Pack class not available');
            }
        } catch (Exception $e) {
            AIOX_Logger::error('Failed to generate AIOX pack file', array('error' => $e->getMessage()));
        }
    }
    
    /**
     * AJAX: Get file status for real-time updates
     */
    public function ajax_get_file_status() {
        check_ajax_referer('aiox_ajax_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }
        
        $well_known_dir = ABSPATH . '.well-known';
        $files = array(
            'aio.json' => 'AIOX Manifest',
            'tdmrep.json' => 'TDM-REP File',
            'aiox-pubkey.pem' => 'Public Key',
            'aiox.ndjson' => 'Delta Feed',
            'aiox.aiox' => 'AIOX Pack'
        );
        
        $file_status = array();
        
        foreach ($files as $filename => $description) {
            $file_path = $well_known_dir . '/' . $filename;
            $exists = file_exists($file_path);
            
            $file_status[$filename] = array(
                'exists' => $exists,
                'size' => $exists ? $this->format_file_size(filesize($file_path)) : null,
                'url' => home_url('/.well-known/' . $filename),
                'description' => $description
            );
        }
        
        wp_send_json_success($file_status);
    }
    
    /**
     * Format file size for display
     */
    private function format_file_size($bytes) {
        if ($bytes >= 1048576) {
            return number_format($bytes / 1048576, 1) . ' MB';
        } elseif ($bytes >= 1024) {
            return number_format($bytes / 1024, 1) . ' KB';
        } else {
            return $bytes . ' B';
        }
    }
    
    /**
     * Add query vars for .well-known file handling
     */
    public function add_query_vars($vars) {
        $vars[] = 'aiox_well_known';
        $vars[] = 'post_id';
        $vars[] = 'aiox_post_id'; // For individual post manifests
        return $vars;
    }
    
    /**
     * Add rewrite rules for .well-known files
     */
    public function add_rewrite_rules() {
        add_rewrite_rule('^\.well-known/ai-info\.json$', 'index.php?aiox_well_known=ai-info', 'top');
        add_rewrite_rule('^\.well-known/ai-info-master\.json$', 'index.php?aiox_well_known=ai-info-master', 'top');
        add_rewrite_rule('^\.well-known/ai-info-([0-9]+)\.json$', 'index.php?aiox_well_known=ai-info-post&post_id=$matches[1]', 'top');
        add_rewrite_rule('^\.well-known/aio\.json$', 'index.php?aiox_well_known=manifest', 'top');
        add_rewrite_rule('^\.well-known/aiox\.aiox$', 'index.php?aiox_well_known=binary-manifest', 'top');
        
        // Individual post manifest files - CRITICAL for AI visibility tools
        // This allows /.well-known/posts/{post_id}.json to work even if static serving fails
        add_rewrite_rule('^\.well-known/posts/([0-9]+)\.json$', 'index.php?aiox_well_known=post-manifest&aiox_post_id=$matches[1]', 'top');
        
        // Also serve aio.json from root for broader AI crawler compatibility
        add_rewrite_rule('^aio\.json$', 'index.php?aiox_well_known=manifest', 'top');
        
        // Note: Rewrite rules are now flushed only during plugin activation
        // to avoid expensive flush_rewrite_rules() on every page load.
        // See aiox_analytics_activate() in main plugin file.
    }
    
    /**
     * Handle .well-known file requests
     */
    public function handle_well_known_requests() {
        // Ensure WordPress is fully initialized before processing
        if (!did_action('wp_loaded')) {
            return;
        }
        
        $well_known = get_query_var('aiox_well_known');
        
        if (empty($well_known)) {
            return;
        }
        
        $well_known_dir = ABSPATH . '.well-known';
        
        switch ($well_known) {
            case 'ai-info':
                $file_path = $well_known_dir . '/ai-info.json';
                $this->serve_json_file($file_path);
                break;
                
            case 'ai-info-master':
                $file_path = $well_known_dir . '/ai-info-master.json';
                $this->serve_json_file($file_path);
                break;
                
            case 'ai-info-post':
                $post_id = get_query_var('post_id');
                if ($post_id) {
                    $file_path = $well_known_dir . '/ai-info-' . $post_id . '.json';
                    $this->serve_json_file($file_path);
                }
                break;
                
            case 'manifest':
                $file_path = $well_known_dir . '/aio.json';
                $this->serve_json_file($file_path);
                break;
                
            case 'binary-manifest':
                $file_path = $well_known_dir . '/aiox.aiox';
                $this->serve_binary_file($file_path);
                break;
                
            case 'post-manifest':
                // Handle individual post manifest files
                // This is CRITICAL for AI visibility tools to access /.well-known/posts/{post_id}.json
                $post_id = absint(get_query_var('aiox_post_id'));
                if ($post_id) {
                    $file_path = $well_known_dir . '/posts/' . $post_id . '.json';
                    
                    // If static file exists, serve it
                    if (file_exists($file_path)) {
                        $this->serve_json_file($file_path);
                    } else {
                        // Generate dynamically if file doesn't exist
                        $post_data = $this->generate_post_manifest_data($post_id);
                        if ($post_data) {
                            header('Content-Type: application/json; charset=utf-8');
                            header('Cache-Control: public, max-age=3600');
                            header('Access-Control-Allow-Origin: *');
                            echo wp_json_encode($post_data, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
                            exit;
                        } else {
                            status_header(404);
                            echo json_encode(array('error' => 'Post not found or not processed'));
                            exit;
                        }
                    }
                }
                break;
        }
    }
    
    /**
     * Generate post manifest data dynamically
     * Used when static file doesn't exist
     */
    private function generate_post_manifest_data($post_id) {
        $post = get_post($post_id);
        
        if (!$post || $post->post_status !== 'publish') {
            return null;
        }
        
        // Check if post has been processed
        $is_processed = get_post_meta($post_id, '_aiox_processed', true);
        if (!$is_processed) {
            return null;
        }
        
        // Get home URL with safety check
        $home_url = home_url();
        if (empty($home_url) || !filter_var($home_url, FILTER_VALIDATE_URL)) {
            // Fallback to site_url if home_url is invalid
            $home_url = site_url();
            if (empty($home_url) || !filter_var($home_url, FILTER_VALIDATE_URL)) {
                return null; // Cannot proceed without a valid URL
            }
        }
        
        // Get ingredients
        $ingredients = array();
        if (class_exists('AIOX_Ingredients')) {
            $ingredients = AIOX_Ingredients::get_post_ingredients($post_id);
        }
        
        // Get metadata
        $metadata = array();
        if (class_exists('AIOX_Metadata')) {
            $metadata = AIOX_Metadata::get_post_metadata($post_id);
        }
        
        // Get capsule data
        $capsule_data = array();
        if (class_exists('AIOX_Capsule_Manager')) {
            $capsule_manager = new AIOX_Capsule_Manager();
            $capsule = $capsule_manager->get_capsule_by_post_id($post_id);
            if ($capsule) {
                $capsule_data = array(
                    'id' => $capsule->get_id(),
                    'status' => $capsule->get_status(),
                    'type' => $capsule->get_type()
                );
            }
        }
        
        // Get license and monetization presets (inherited from site settings)
        $license_config = self::get_license_preset_config();
        $monetization_config = self::get_monetization_preset_config();
        $enforcement_config = self::get_enforcement_mode_config();
        
        // Check for post-level overrides (future feature)
        $post_license_override = get_post_meta($post_id, '_aiox_license_override', true);
        $post_monetization_override = get_post_meta($post_id, '_aiox_monetization_override', true);
        
        return array(
            'version' => '2.0',
            'generator' => 'AIOX Publisher Suite Pro',
            'generated_at' => current_time('c'),
            'post' => array(
                'id' => $post_id,
                'title' => $post->post_title,
                'url' => get_permalink($post_id),
                'published' => get_the_date('c', $post_id),
                'modified' => get_the_modified_date('c', $post_id),
                'type' => $post->post_type,
                'author' => get_the_author_meta('display_name', $post->post_author),
                'excerpt' => wp_trim_words(get_the_excerpt($post_id), 50, '...'),
                'word_count' => str_word_count(strip_tags($post->post_content)),
                'categories' => wp_get_post_categories($post_id, array('fields' => 'names')),
                'tags' => wp_get_post_tags($post_id, array('fields' => 'names'))
            ),
            'licensing' => array(
                'preset' => $post_license_override ?: $license_config['preset'],
                'license_code' => $license_config['license_code'],
                'training_allowed' => $license_config['training_allowed'],
                'citation_required' => $license_config['citation_required'],
                'attribution_required' => $license_config['attribution_required'],
                'commercial_use' => $license_config['commercial_use'],
                'attribution' => get_option('aiox_attribution_string', get_bloginfo('name')),
                'inherited_from' => $post_license_override ? 'post' : 'site'
            ),
            'monetization' => array(
                'preset' => $post_monetization_override ?: $monetization_config['preset'],
                'model' => $monetization_config['pricing_model'],
                'subscription_required' => $monetization_config['subscription_required'],
                'pay_per_use' => $monetization_config['pay_per_use'],
                'inherited_from' => $post_monetization_override ? 'post' : 'site'
            ),
            'enforcement' => array(
                'mode' => $enforcement_config['mode'],
                'compliance_level' => $enforcement_config['compliance_level'],
                'legal_binding' => $enforcement_config['legal_binding']
            ),
            'ingredients' => array(
                'count' => count($ingredients),
                'items' => $ingredients
            ),
            'capsule' => $capsule_data,
            'metadata' => $metadata,
            'processing' => array(
                'processed' => true,
                'last_processed' => get_post_meta($post_id, '_aiox_processed_at', true),
                'server_processed' => (bool) get_post_meta($post_id, '_aiox_server_processed', true)
            ),
            'links' => array(
                'self' => $home_url . '/.well-known/posts/' . $post_id . '.json',
                'post' => get_permalink($post_id),
                'manifest' => $home_url . '/.well-known/aio.json',
                'tdmrep' => $home_url . '/.well-known/tdmrep.json'
            )
        );
    }
    
    /**
     * Serve JSON file with proper headers
     */
    private function serve_json_file($file_path) {
        if (!file_exists($file_path)) {
            status_header(404);
            exit('File not found');
        }
        
        header('Content-Type: application/json; charset=utf-8');
        header('Cache-Control: public, max-age=3600');
        header('Access-Control-Allow-Origin: *');
        
        readfile($file_path);
        exit;
    }
    
    /**
     * Serve binary file with proper headers
     */
    private function serve_binary_file($file_path) {
        if (!file_exists($file_path)) {
            status_header(404);
            exit('File not found');
        }
        
        header('Content-Type: application/octet-stream');
        header('Cache-Control: public, max-age=3600');
        header('Access-Control-Allow-Origin: *');
        
        readfile($file_path);
        exit;
    }
    
    /**
     * Process queue (cron job)
     */
    public function process_queue() {
        AIOX_Logger::info('Queue processing started');
        
        global $wpdb;
        $table_name = $wpdb->prefix . 'aiox_queue';
        
        // Get pending items from queue
        $queue_items = $wpdb->get_results(
            "SELECT * FROM {$table_name} WHERE status = 'pending' ORDER BY priority DESC, created_at ASC LIMIT 10",
            ARRAY_A
        );
        
        if (empty($queue_items)) {
            AIOX_Logger::debug('No items in queue to process');
            return;
        }
        
        // ✅ Use Comprehensive Processor instead of Generator
        require_once AIOX_PLUGIN_DIR . 'core/class-aiox-comprehensive-processor.php';
        $processor = new AIOX_Comprehensive_Processor();
        $processed = 0;
        
        foreach ($queue_items as $item) {
            // Update status to processing
            $wpdb->update(
                $table_name,
                array('status' => 'processing', 'updated_at' => current_time('mysql')),
                array('id' => $item['id']),
                array('%s', '%s'),
                array('%d')
            );
            
            // Process using comprehensive processor (API-only)
            try {
                $result = $processor->process_post_completely($item['post_id']);
                $success = isset($result['success']) ? $result['success'] : false;
                
                if ($success) {
                    // Remove from queue on success
                    $wpdb->delete($table_name, array('id' => $item['id']), array('%d'));
                    $processed++;
                    
                    AIOX_Logger::info("Queue item processed successfully", array(
                        'post_id' => $item['post_id'],
                        'capsule_id' => isset($result['capsule_id']) ? $result['capsule_id'] : null
                    ));
                } else {
                    // Mark as failed
                    $wpdb->update(
                        $table_name,
                        array('status' => 'failed', 'updated_at' => current_time('mysql')),
                        array('id' => $item['id']),
                        array('%s', '%s'),
                        array('%d')
                    );
                    
                    AIOX_Logger::error("Queue item failed", array(
                        'post_id' => $item['post_id'],
                        'message' => isset($result['message']) ? $result['message'] : 'Unknown error'
                    ));
                }
            } catch (Exception $e) {
                // Mark as failed on exception
                $wpdb->update(
                    $table_name,
                    array('status' => 'failed', 'updated_at' => current_time('mysql')),
                    array('id' => $item['id']),
                    array('%s', '%s'),
                    array('%d')
                );
                
                AIOX_Logger::error("Queue processing exception", array(
                    'post_id' => $item['post_id'],
                    'error' => $e->getMessage()
                ));
            }
        }
        
        AIOX_Logger::info("Queue processing completed", array(
            'processed' => $processed,
            'total_items' => count($queue_items)
        ));
        
        AIOX_Utils::update_option('last_queue_process', current_time('mysql'));
    }
    
    /**
     * Cleanup logs (cron job)
     */
    public function cleanup_logs() {
        $retention_days = AIOX_Utils::get_option('log_retention_days', 30);
        
        global $wpdb;
        $table_name = $wpdb->prefix . 'aiox_logs';
        $cutoff_date = date('Y-m-d H:i:s', strtotime("-{$retention_days} days"));
        
        $deleted = $wpdb->query($wpdb->prepare(
            "DELETE FROM {$table_name} WHERE created_at < %s",
            $cutoff_date
        ));
        
        if ($deleted > 0) {
            AIOX_Logger::info("Cleaned up {$deleted} old log entries");
        }
    }
    
    /**
     * Register additional REST routes
     */
    public function register_rest_routes() {
        register_rest_route('aiox/v1', '/status', array(
            'methods' => 'GET',
            'callback' => array($this, 'get_status'),
            'permission_callback' => '__return_true'
        ));
    }
    
    /**
     * Get plugin status via REST API
     */
    public function get_status() {
        return rest_ensure_response(AIOX_Utils::get_plugin_status());
    }
    
    /**
     * Add to processing queue
     */
    private function add_to_queue($post_id, $action = 'update') {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'aiox_queue';
        
        // Check if table exists
        if ($wpdb->get_var("SHOW TABLES LIKE '{$table_name}'") !== $table_name) {
            return; // Table doesn't exist yet, skip
        }
        
        // Check if already in queue
        $existing = $wpdb->get_var($wpdb->prepare(
            "SELECT id FROM {$table_name} WHERE post_id = %d AND status IN ('pending', 'processing')",
            $post_id
        ));
        
        if ($existing) {
            // Update existing entry
            $wpdb->update(
                $table_name,
                array(
                    'status' => 'pending',
                    'updated_at' => current_time('mysql')
                ),
                array('id' => $existing),
                array('%s', '%s'),
                array('%d')
            );
        } else {
            // Insert new entry
            $wpdb->insert(
                $table_name,
                array(
                    'post_id' => $post_id,
                    'status' => 'pending',
                    'priority' => 10,
                    'created_at' => current_time('mysql'),
                    'updated_at' => current_time('mysql')
                ),
                array('%d', '%s', '%d', '%s', '%s')
            );
        }
    }
    
    /**
     * Remove from processing queue
     */
    private function remove_from_queue($post_id) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'aiox_queue';
        
        // Check if table exists
        if ($wpdb->get_var("SHOW TABLES LIKE '{$table_name}'") !== $table_name) {
            return; // Table doesn't exist yet, skip
        }
        
        $wpdb->delete(
            $table_name,
            array('post_id' => $post_id),
            array('%d')
        );
    }
    
    /**
     * Cleanup post files
     */
    private function cleanup_post_files($post_id) {
        // Clean up any generated files for this post
        // Will be implemented in later phases when file generation is added
        AIOX_Logger::debug('Cleaning up files for post', array('post_id' => $post_id));
    }
    
    /**
     * Add structured data
     * 
     * NOTE: This method is now disabled in favor of AIOX_Enhanced_Microdata class
     * which provides more comprehensive and properly formatted Schema.org output.
     * Keeping this code for reference but not executing to avoid duplication.
     */
    private function add_structured_data() {
        // Disabled - AIOX_Enhanced_Microdata class now handles all Schema.org output
        return;
        
        /* ORIGINAL CODE PRESERVED FOR REFERENCE:
        global $post;
        $post_id = $post->ID;
        
        // Get processed data
        $processed = get_post_meta($post_id, '_aiox_processed_data', true);
        $processing_method = 'unknown';
        
        if (!empty($processed)) {
            $processed_array = is_string($processed) ? json_decode($processed, true) : $processed;
            if (isset($processed_array['processing_method'])) {
                $processing_method = $processed_array['processing_method'];
            }
        }
        */
    }
    
    /**
     * Add AI badge
     */
    private function add_ai_badge() {
        // Badge is now handled by AIOX_Badge class - do nothing here to avoid duplicates
        return;
    }
    
    /**
 * Get lightweight content index for manifest
 * NEW: Returns minimal data with links to individual post files
 */
private static function get_content_data_with_working_ingredients() {
    // Get posts that have been processed
    $posts = get_posts(array(
        'post_type' => array('post', 'page'),
        'post_status' => 'publish',
        'numberposts' => -1,
        'meta_query' => array(
            array(
                'key' => '_aiox_processed',
                'value' => '1',
                'compare' => '='
            )
        )
    ));
    
    $content_index = array();
    $home_url = home_url();
    
    foreach ($posts as $post) {
        // Get counts only - don't load full data
        $ingredients_data = get_post_meta($post->ID, '_aiox_ingredients', true);
        $ingredients_array = array();
        if (!empty($ingredients_data)) {
            $decoded = json_decode($ingredients_data, true);
            if (is_array($decoded)) {
                $ingredients_array = $decoded;
            }
        }
        
        $metadata_data = get_post_meta($post->ID, '_aiox_metadata', true);
        $metadata = array();
        if (!empty($metadata_data)) {
            $decoded = json_decode($metadata_data, true);
            if (is_array($decoded)) {
                $metadata = $decoded;
            }
        }
        
        // Create lightweight index entry
        $index_entry = array(
            'id' => $post->ID,
            'title' => $post->post_title,
            'url' => get_permalink($post->ID),
            'type' => $post->post_type,
            'published_at' => get_the_date('c', $post->ID),
            'modified_at' => get_the_modified_date('c', $post->ID),
            'author' => get_the_author_meta('display_name', $post->post_author),
            'excerpt' => wp_trim_words(get_the_excerpt($post->ID), 30, '...'),
            'word_count' => str_word_count(strip_tags($post->post_content)),
            'reading_time' => ceil(str_word_count(strip_tags($post->post_content)) / 200),
            'summary' => array(
                'ingredients_count' => count($ingredients_array),
                'metadata_fields' => count($metadata),
                'categories' => wp_get_post_categories($post->ID, array('fields' => 'names')),
                'tags' => array_slice(wp_get_post_tags($post->ID, array('fields' => 'names')), 0, 5)
            ),
            'detail_url' => $home_url . '/.well-known/posts/' . $post->ID . '.json'
        );
        
        $content_index[] = $index_entry;
    }
    
    return $content_index;
}
    
    /**
     * Check for plugin updates
     */
    private function check_plugin_updates() {
        $current_version = AIOX_Utils::get_option('version');
        
        if ($current_version !== AIOX_VERSION) {
            // Plugin was updated
            AIOX_Utils::update_option('version', AIOX_VERSION);
            AIOX_Logger::info('Plugin updated', array(
                'old_version' => $current_version,
                'new_version' => AIOX_VERSION
            ));
        }
    }
}