<?php
/**
 * AIOX Logs Viewer
 * 
 * Admin interface for viewing and managing AIOX logs
 */

if (!defined('ABSPATH')) {
    exit;
}

class AIOX_Logs_Viewer {
    
    /**
     * Initialize the logs viewer
     */
    public function __construct() {
        add_action('wp_ajax_aiox_get_logs', array($this, 'ajax_get_logs'));
        add_action('wp_ajax_aiox_delete_logs', array($this, 'ajax_delete_logs'));
        add_action('wp_ajax_aiox_clear_all_logs', array($this, 'ajax_clear_all_logs'));
        add_action('wp_ajax_aiox_delete_old_logs', array($this, 'ajax_delete_old_logs'));
        add_action('wp_ajax_aiox_export_logs', array($this, 'ajax_export_logs'));
    }
    
    /**
     * Render logs viewer section
     */
    public function render_logs_viewer() {
        $stats = AIOX_Logger::get_statistics();
        $categories = AIOX_Logger::get_categories();
        ?>
        <div class="aiox-diagnostics-section">
            <h2 class="aiox-section-title">
                <span class="dashicons dashicons-list-view"></span>
                <?php _e('System Logs', 'aiox-publisher-suite-pro'); ?>
            </h2>
            
            <!-- Statistics -->
            <div class="aiox-logs-stats">
                <div class="aiox-stat-card">
                    <div class="stat-label"><?php _e('Total Logs', 'aiox-publisher-suite-pro'); ?></div>
                    <div class="stat-value"><?php echo number_format($stats['total']); ?></div>
                </div>
                <div class="aiox-stat-card warning">
                    <div class="stat-label"><?php _e('Recent Errors (24h)', 'aiox-publisher-suite-pro'); ?></div>
                    <div class="stat-value"><?php echo number_format($stats['recent_errors']); ?></div>
                </div>
                <?php if (!empty($stats['by_level'])): ?>
                    <?php foreach ($stats['by_level'] as $level => $count): ?>
                        <div class="aiox-stat-card <?php echo esc_attr($level); ?>">
                            <div class="stat-label"><?php echo ucfirst($level); ?></div>
                            <div class="stat-value"><?php echo number_format($count); ?></div>
                        </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>
            
            <!-- Filters -->
            <div class="aiox-logs-filters">
                <div class="filter-row">
                    <div class="filter-group">
                        <label><?php _e('Level:', 'aiox-publisher-suite-pro'); ?></label>
                        <select id="aiox-filter-level" class="aiox-filter">
                            <option value=""><?php _e('All Levels', 'aiox-publisher-suite-pro'); ?></option>
                            <option value="debug"><?php _e('Debug', 'aiox-publisher-suite-pro'); ?></option>
                            <option value="info"><?php _e('Info', 'aiox-publisher-suite-pro'); ?></option>
                            <option value="warning"><?php _e('Warning', 'aiox-publisher-suite-pro'); ?></option>
                            <option value="error"><?php _e('Error', 'aiox-publisher-suite-pro'); ?></option>
                            <option value="critical"><?php _e('Critical', 'aiox-publisher-suite-pro'); ?></option>
                        </select>
                    </div>
                    
                    <div class="filter-group">
                        <label><?php _e('Category:', 'aiox-publisher-suite-pro'); ?></label>
                        <select id="aiox-filter-category" class="aiox-filter">
                            <option value=""><?php _e('All Categories', 'aiox-publisher-suite-pro'); ?></option>
                            <?php foreach ($categories as $category): ?>
                                <option value="<?php echo esc_attr($category); ?>"><?php echo esc_html($category); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div class="filter-group">
                        <label><?php _e('Post ID:', 'aiox-publisher-suite-pro'); ?></label>
                        <input type="number" id="aiox-filter-post-id" class="aiox-filter" placeholder="<?php _e('Filter by post', 'aiox-publisher-suite-pro'); ?>">
                    </div>
                    
                    <div class="filter-group">
                        <label><?php _e('Search:', 'aiox-publisher-suite-pro'); ?></label>
                        <input type="text" id="aiox-filter-search" class="aiox-filter" placeholder="<?php _e('Search messages', 'aiox-publisher-suite-pro'); ?>">
                    </div>
                </div>
                
                <div class="filter-row">
                    <div class="filter-group">
                        <label><?php _e('Date From:', 'aiox-publisher-suite-pro'); ?></label>
                        <input type="date" id="aiox-filter-date-from" class="aiox-filter">
                    </div>
                    
                    <div class="filter-group">
                        <label><?php _e('Date To:', 'aiox-publisher-suite-pro'); ?></label>
                        <input type="date" id="aiox-filter-date-to" class="aiox-filter">
                    </div>
                    
                    <div class="filter-actions">
                        <button type="button" id="aiox-apply-filters" class="button button-primary">
                            <span class="dashicons dashicons-filter"></span>
                            <?php _e('Apply Filters', 'aiox-publisher-suite-pro'); ?>
                        </button>
                        <button type="button" id="aiox-reset-filters" class="button">
                            <span class="dashicons dashicons-update"></span>
                            <?php _e('Reset', 'aiox-publisher-suite-pro'); ?>
                        </button>
                    </div>
                </div>
            </div>
            
            <!-- Actions -->
            <div class="aiox-logs-actions">
                <button type="button" id="aiox-delete-selected-logs" class="button" disabled>
                    <span class="dashicons dashicons-trash"></span>
                    <?php _e('Delete Selected', 'aiox-publisher-suite-pro'); ?>
                </button>
                <button type="button" id="aiox-export-logs" class="button">
                    <span class="dashicons dashicons-download"></span>
                    <?php _e('Export', 'aiox-publisher-suite-pro'); ?>
                </button>
                <button type="button" id="aiox-delete-old-logs" class="button">
                    <span class="dashicons dashicons-calendar"></span>
                    <?php _e('Delete Logs Older Than 30 Days', 'aiox-publisher-suite-pro'); ?>
                </button>
                <button type="button" id="aiox-clear-all-logs" class="button button-danger">
                    <span class="dashicons dashicons-warning"></span>
                    <?php _e('Clear All Logs', 'aiox-publisher-suite-pro'); ?>
                </button>
            </div>
            
            <!-- Logs Table -->
            <div class="aiox-logs-container">
                <table class="aiox-logs-table">
                    <thead>
                        <tr>
                            <th class="check-column">
                                <input type="checkbox" id="aiox-select-all-logs">
                            </th>
                            <th><?php _e('Timestamp', 'aiox-publisher-suite-pro'); ?></th>
                            <th><?php _e('Level', 'aiox-publisher-suite-pro'); ?></th>
                            <th><?php _e('Category', 'aiox-publisher-suite-pro'); ?></th>
                            <th><?php _e('Message', 'aiox-publisher-suite-pro'); ?></th>
                            <th><?php _e('Post ID', 'aiox-publisher-suite-pro'); ?></th>
                            <th><?php _e('Actions', 'aiox-publisher-suite-pro'); ?></th>
                        </tr>
                    </thead>
                    <tbody id="aiox-logs-tbody">
                        <tr>
                            <td colspan="7" class="loading"><?php _e('Loading logs...', 'aiox-publisher-suite-pro'); ?></td>
                        </tr>
                    </tbody>
                </table>
            </div>
            
            <!-- Pagination -->
            <div class="aiox-logs-pagination">
                <div class="pagination-info">
                    <span id="aiox-logs-info"></span>
                </div>
                <div class="pagination-controls" id="aiox-pagination-controls"></div>
            </div>
        </div>
        
        <!-- Log Detail Modal -->
        <div id="aiox-log-detail-modal" class="aiox-modal" style="display:none;">
            <div class="aiox-modal-content">
                <span class="aiox-modal-close">&times;</span>
                <h3><?php _e('Log Details', 'aiox-publisher-suite-pro'); ?></h3>
                <div id="aiox-log-detail-content"></div>
            </div>
        </div>
        
        <style>
        .aiox-logs-stats {
            display: flex;
            gap: 15px;
            margin-bottom: 20px;
            flex-wrap: wrap;
        }
        
        .aiox-stat-card {
            background: #fff;
            border: 1px solid #ddd;
            border-radius: 6px;
            padding: 15px 20px;
            min-width: 140px;
            box-shadow: 0 1px 3px rgba(0,0,0,0.05);
        }
        
        .aiox-stat-card.warning {
            border-left: 4px solid #f0ad4e;
        }
        
        .aiox-stat-card.error, .aiox-stat-card.critical {
            border-left: 4px solid #d9534f;
        }
        
        .aiox-stat-card.debug {
            border-left: 4px solid #5bc0de;
        }
        
        .aiox-stat-card.info {
            border-left: 4px solid #5cb85c;
        }
        
        .stat-label {
            font-size: 12px;
            color: #666;
            margin-bottom: 5px;
        }
        
        .stat-value {
            font-size: 24px;
            font-weight: bold;
            color: #333;
        }
        
        .aiox-logs-filters {
            background: #f8f9fa;
            padding: 20px;
            border-radius: 6px;
            margin-bottom: 20px;
        }
        
        .filter-row {
            display: flex;
            gap: 15px;
            margin-bottom: 15px;
            flex-wrap: wrap;
        }
        
        .filter-row:last-child {
            margin-bottom: 0;
        }
        
        .filter-group {
            display: flex;
            flex-direction: column;
            gap: 5px;
            flex: 1;
            min-width: 150px;
        }
        
        .filter-group label {
            font-size: 12px;
            font-weight: 600;
            color: #555;
        }
        
        .filter-group input,
        .filter-group select {
            padding: 6px 10px;
            border: 1px solid #ddd;
            border-radius: 4px;
        }
        
        .filter-actions {
            display: flex;
            gap: 10px;
            align-items: flex-end;
        }
        
        .aiox-logs-actions {
            display: flex;
            gap: 10px;
            margin-bottom: 15px;
            flex-wrap: wrap;
        }
        
        .button-danger {
            background: #d9534f;
            border-color: #d43f3a;
            color: #fff;
        }
        
        .button-danger:hover {
            background: #c9302c;
            border-color: #ac2925;
        }
        
        .aiox-logs-container {
            background: #fff;
            border: 1px solid #ddd;
            border-radius: 6px;
            overflow-x: auto;
            margin-bottom: 20px;
        }
        
        .aiox-logs-table {
            width: 100%;
            border-collapse: collapse;
        }
        
        .aiox-logs-table th,
        .aiox-logs-table td {
            padding: 12px;
            text-align: left;
            border-bottom: 1px solid #eee;
        }
        
        .aiox-logs-table th {
            background: #f8f9fa;
            font-weight: 600;
            color: #555;
            position: sticky;
            top: 0;
            z-index: 10;
        }
        
        .aiox-logs-table tr:hover {
            background: #f8f9fa;
        }
        
        .check-column {
            width: 40px;
        }
        
        .log-level {
            display: inline-block;
            padding: 4px 8px;
            border-radius: 3px;
            font-size: 11px;
            font-weight: 600;
            text-transform: uppercase;
        }
        
        .log-level.debug { background: #d1ecf1; color: #0c5460; }
        .log-level.info { background: #d4edda; color: #155724; }
        .log-level.warning { background: #fff3cd; color: #856404; }
        .log-level.error { background: #f8d7da; color: #721c24; }
        .log-level.critical { background: #d9534f; color: #fff; }
        
        .log-message {
            max-width: 400px;
            overflow: hidden;
            text-overflow: ellipsis;
            white-space: nowrap;
            cursor: pointer;
        }
        
        .log-message:hover {
            text-decoration: underline;
        }
        
        .aiox-logs-pagination {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 15px 0;
        }
        
        .pagination-controls {
            display: flex;
            gap: 5px;
        }
        
        .pagination-controls button {
            padding: 6px 12px;
            border: 1px solid #ddd;
            background: #fff;
            cursor: pointer;
            border-radius: 4px;
        }
        
        .pagination-controls button:hover {
            background: #f8f9fa;
        }
        
        .pagination-controls button.active {
            background: #0073aa;
            color: #fff;
            border-color: #0073aa;
        }
        
        .pagination-controls button:disabled {
            opacity: 0.5;
            cursor: not-allowed;
        }
        
        .aiox-modal {
            display: none;
            position: fixed;
            z-index: 100000;
            left: 0;
            top: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(0,0,0,0.5);
        }
        
        .aiox-modal-content {
            background-color: #fff;
            margin: 50px auto;
            padding: 30px;
            border: 1px solid #ddd;
            width: 80%;
            max-width: 800px;
            border-radius: 8px;
            box-shadow: 0 4px 20px rgba(0,0,0,0.2);
            max-height: 80vh;
            overflow-y: auto;
        }
        
        .aiox-modal-close {
            color: #aaa;
            float: right;
            font-size: 28px;
            font-weight: bold;
            cursor: pointer;
        }
        
        .aiox-modal-close:hover {
            color: #000;
        }
        
        #aiox-log-detail-content {
            margin-top: 20px;
        }
        
        .log-detail-field {
            margin-bottom: 15px;
            padding: 10px;
            background: #f8f9fa;
            border-radius: 4px;
        }
        
        .log-detail-label {
            font-weight: 600;
            color: #555;
            margin-bottom: 5px;
        }
        
        .log-detail-value {
            color: #333;
            word-break: break-word;
        }
        
        .log-context-json {
            background: #2d2d2d;
            color: #f8f8f2;
            padding: 15px;
            border-radius: 4px;
            font-family: 'Courier New', monospace;
            font-size: 12px;
            overflow-x: auto;
        }
        </style>
        
        <script>
        jQuery(document).ready(function($) {
            let currentPage = 1;
            let currentFilters = {};
            
            // Load logs
            function loadLogs(page = 1) {
                currentPage = page;
                
                $('#aiox-logs-tbody').html('<tr><td colspan="7" class="loading">Loading logs...</td></tr>');
                
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'aiox_get_logs',
                        nonce: '<?php echo wp_create_nonce("aiox_logs_nonce"); ?>',
                        page: page,
                        filters: currentFilters
                    },
                    success: function(response) {
                        if (response.success) {
                            renderLogs(response.data);
                        } else {
                            $('#aiox-logs-tbody').html('<tr><td colspan="7" class="error">' + response.data.message + '</td></tr>');
                        }
                    }
                });
            }
            
            // Render logs table
            function renderLogs(data) {
                let html = '';
                
                if (data.logs.length === 0) {
                    html = '<tr><td colspan="7" style="text-align:center;padding:20px;">No logs found</td></tr>';
                } else {
                    data.logs.forEach(function(log) {
                        html += '<tr>';
                        html += '<td><input type="checkbox" class="log-checkbox" value="' + log.id + '"></td>';
                        html += '<td>' + log.timestamp + '</td>';
                        html += '<td><span class="log-level ' + log.level + '">' + log.level + '</span></td>';
                        html += '<td>' + log.category + '</td>';
                        html += '<td class="log-message" data-log-id="' + log.id + '">' + escapeHtml(log.message) + '</td>';
                        html += '<td>' + (log.post_id ? '<a href="post.php?post=' + log.post_id + '&action=edit">' + log.post_id + '</a>' : '-') + '</td>';
                        html += '<td><button class="button button-small view-log-detail" data-log-id="' + log.id + '">View</button></td>';
                        html += '</tr>';
                    });
                }
                
                $('#aiox-logs-tbody').html(html);
                
                // Update pagination
                renderPagination(data);
                
                // Update info
                let start = (data.current_page - 1) * 50 + 1;
                let end = Math.min(data.current_page * 50, data.total);
                $('#aiox-logs-info').text('Showing ' + start + ' to ' + end + ' of ' + data.total + ' logs');
            }
            
            // Render pagination
            function renderPagination(data) {
                let html = '';
                
                // Previous button
                html += '<button ' + (data.current_page === 1 ? 'disabled' : '') + ' data-page="' + (data.current_page - 1) + '">« Prev</button>';
                
                // Page numbers
                let startPage = Math.max(1, data.current_page - 2);
                let endPage = Math.min(data.pages, data.current_page + 2);
                
                if (startPage > 1) {
                    html += '<button data-page="1">1</button>';
                    if (startPage > 2) html += '<span>...</span>';
                }
                
                for (let i = startPage; i <= endPage; i++) {
                    html += '<button ' + (i === data.current_page ? 'class="active"' : '') + ' data-page="' + i + '">' + i + '</button>';
                }
                
                if (endPage < data.pages) {
                    if (endPage < data.pages - 1) html += '<span>...</span>';
                    html += '<button data-page="' + data.pages + '">' + data.pages + '</button>';
                }
                
                // Next button
                html += '<button ' + (data.current_page === data.pages ? 'disabled' : '') + ' data-page="' + (data.current_page + 1) + '">Next »</button>';
                
                $('#aiox-pagination-controls').html(html);
            }
            
            // Pagination click
            $(document).on('click', '#aiox-pagination-controls button', function() {
                let page = parseInt($(this).data('page'));
                if (page > 0) {
                    loadLogs(page);
                }
            });
            
            // Apply filters
            $('#aiox-apply-filters').on('click', function() {
                currentFilters = {
                    level: $('#aiox-filter-level').val(),
                    category: $('#aiox-filter-category').val(),
                    post_id: $('#aiox-filter-post-id').val(),
                    search: $('#aiox-filter-search').val(),
                    date_from: $('#aiox-filter-date-from').val(),
                    date_to: $('#aiox-filter-date-to').val()
                };
                loadLogs(1);
            });
            
            // Reset filters
            $('#aiox-reset-filters').on('click', function() {
                $('.aiox-filter').val('');
                currentFilters = {};
                loadLogs(1);
            });
            
            // Select all
            $('#aiox-select-all-logs').on('change', function() {
                $('.log-checkbox').prop('checked', $(this).prop('checked'));
                updateDeleteButton();
            });
            
            // Update delete button
            $(document).on('change', '.log-checkbox', updateDeleteButton);
            
            function updateDeleteButton() {
                let checked = $('.log-checkbox:checked').length;
                $('#aiox-delete-selected-logs').prop('disabled', checked === 0);
            }
            
            // Delete selected
            $('#aiox-delete-selected-logs').on('click', function() {
                if (!confirm('Are you sure you want to delete the selected logs?')) return;
                
                let ids = [];
                $('.log-checkbox:checked').each(function() {
                    ids.push($(this).val());
                });
                
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'aiox_delete_logs',
                        nonce: '<?php echo wp_create_nonce("aiox_logs_nonce"); ?>',
                        ids: ids
                    },
                    success: function(response) {
                        if (response.success) {
                            loadLogs(currentPage);
                            alert('Logs deleted successfully');
                        } else {
                            alert('Error: ' + response.data.message);
                        }
                    }
                });
            });
            
            // Clear all logs
            $('#aiox-clear-all-logs').on('click', function() {
                if (!confirm('Are you sure you want to delete ALL logs? This cannot be undone!')) return;
                
                var $btn = $(this);
                $btn.prop('disabled', true).text('Clearing...');
                
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'aiox_clear_all_logs',
                        nonce: '<?php echo wp_create_nonce("aiox_logs_nonce"); ?>'
                    },
                    success: function(response) {
                        if (response.success) {
                            loadLogs(1);
                            alert(response.data.message || 'All logs cleared successfully');
                        } else {
                            alert('Error: ' + (response.data.message || 'Unknown error'));
                        }
                    },
                    error: function(xhr, status, error) {
                        alert('AJAX Error: ' + error);
                    },
                    complete: function() {
                        $btn.prop('disabled', false).html('<span class="dashicons dashicons-warning"></span> Clear All Logs');
                    }
                });
            });
            
            // Delete old logs
            $('#aiox-delete-old-logs').on('click', function() {
                if (!confirm('Are you sure you want to delete logs older than 30 days?')) return;
                
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'aiox_delete_old_logs',
                        nonce: '<?php echo wp_create_nonce("aiox_logs_nonce"); ?>'
                    },
                    success: function(response) {
                        if (response.success) {
                            loadLogs(currentPage);
                            alert(response.data.message);
                        } else {
                            alert('Error: ' + response.data.message);
                        }
                    }
                });
            });
            
            // View log detail
            $(document).on('click', '.view-log-detail, .log-message', function() {
                let logId = $(this).data('log-id');
                let log = null;
                
                // Find log in current data
                $('#aiox-logs-tbody tr').each(function() {
                    if ($(this).find('[data-log-id="' + logId + '"]').length > 0) {
                        // Build log object from row data
                        let $row = $(this);
                        log = {
                            id: logId,
                            timestamp: $row.find('td:eq(1)').text(),
                            level: $row.find('.log-level').text(),
                            category: $row.find('td:eq(3)').text(),
                            message: $row.find('.log-message').text(),
                            post_id: $row.find('td:eq(5)').text()
                        };
                    }
                });
                
                if (log) {
                    showLogDetail(log);
                }
            });
            
            // Show log detail modal
            function showLogDetail(log) {
                let html = '';
                html += '<div class="log-detail-field"><div class="log-detail-label">ID:</div><div class="log-detail-value">' + log.id + '</div></div>';
                html += '<div class="log-detail-field"><div class="log-detail-label">Timestamp:</div><div class="log-detail-value">' + log.timestamp + '</div></div>';
                html += '<div class="log-detail-field"><div class="log-detail-label">Level:</div><div class="log-detail-value"><span class="log-level ' + log.level + '">' + log.level + '</span></div></div>';
                html += '<div class="log-detail-field"><div class="log-detail-label">Category:</div><div class="log-detail-value">' + log.category + '</div></div>';
                html += '<div class="log-detail-field"><div class="log-detail-label">Message:</div><div class="log-detail-value">' + escapeHtml(log.message) + '</div></div>';
                if (log.post_id && log.post_id !== '-') {
                    html += '<div class="log-detail-field"><div class="log-detail-label">Post ID:</div><div class="log-detail-value">' + log.post_id + '</div></div>';
                }
                
                $('#aiox-log-detail-content').html(html);
                $('#aiox-log-detail-modal').show();
            }
            
            // Close modal
            $('.aiox-modal-close').on('click', function() {
                $('#aiox-log-detail-modal').hide();
            });
            
            $(window).on('click', function(event) {
                if (event.target.id === 'aiox-log-detail-modal') {
                    $('#aiox-log-detail-modal').hide();
                }
            });
            
            // Helper function
            function escapeHtml(text) {
                let map = {
                    '&': '&amp;',
                    '<': '&lt;',
                    '>': '&gt;',
                    '"': '&quot;',
                    "'": '&#039;'
                };
                return text.replace(/[&<>"']/g, function(m) { return map[m]; });
            }
            
            // Initial load
            loadLogs(1);
        });
        </script>
        <?php
    }
    
    /**
     * AJAX: Get logs
     */
    public function ajax_get_logs() {
        check_ajax_referer('aiox_logs_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Insufficient permissions'));
        }
        
        $page = isset($_POST['page']) ? intval($_POST['page']) : 1;
        $filters = isset($_POST['filters']) ? $_POST['filters'] : array();
        
        $args = array_merge($filters, array('page' => $page));
        $result = AIOX_Logger::get_logs($args);
        
        wp_send_json_success($result);
    }
    
    /**
     * AJAX: Delete logs
     */
    public function ajax_delete_logs() {
        check_ajax_referer('aiox_logs_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Insufficient permissions'));
        }
        
        $ids = isset($_POST['ids']) ? $_POST['ids'] : array();
        
        if (empty($ids)) {
            wp_send_json_error(array('message' => 'No logs selected'));
        }
        
        $deleted = AIOX_Logger::delete_logs($ids);
        
        if ($deleted) {
            wp_send_json_success(array('message' => $deleted . ' logs deleted'));
        } else {
            wp_send_json_error(array('message' => 'Failed to delete logs'));
        }
    }
    
    /**
     * AJAX: Clear all logs
     */
    public function ajax_clear_all_logs() {
        check_ajax_referer('aiox_logs_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Insufficient permissions'));
        }
        
        $result = AIOX_Logger::clear_all_logs();
        
        if (is_array($result) && isset($result['success'])) {
            if ($result['success']) {
                $deleted = isset($result['deleted']) ? $result['deleted'] : 0;
                wp_send_json_success(array('message' => "Successfully deleted {$deleted} log entries"));
            } else {
                $error_msg = isset($result['error']) ? $result['error'] : 'Unknown error';
                wp_send_json_error(array('message' => 'Failed to clear logs: ' . $error_msg));
            }
        } else {
            // Fallback for old return format
            if ($result) {
                wp_send_json_success(array('message' => 'All logs cleared'));
            } else {
                wp_send_json_error(array('message' => 'Failed to clear logs'));
            }
        }
    }
    
    /**
     * AJAX: Delete old logs
     */
    public function ajax_delete_old_logs() {
        check_ajax_referer('aiox_logs_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Insufficient permissions'));
        }
        
        $deleted = AIOX_Logger::delete_old_logs(30);
        wp_send_json_success(array('message' => $deleted . ' old logs deleted'));
    }
}