<?php
/**
 * AIOX Phase 1 Activation Handler
 * 
 * Handles Phase 1 specific activation tasks without interfering with
 * existing activation functionality.
 */

if (!defined('ABSPATH')) {
    exit;
}

class AIOX_Phase1_Activation {
    
    /**
     * Run Phase 1 activation tasks
     */
    public static function activate() {
        try {
            AIOX_Logger::info('AIOX Phase 1: Starting activation tasks');
            
            // Set Phase 1 default options
            self::set_phase1_default_options();
            
            // Initialize Phase 1 database options
            self::initialize_phase1_database();
            
            // Schedule Phase 1 cleanup tasks
            self::schedule_phase1_tasks();
            
            AIOX_Logger::info('AIOX Phase 1: Activation tasks completed successfully');
            
        } catch (Exception $e) {
            AIOX_Logger::error('AIOX Phase 1: Activation failed', array(
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ));
            
            // Don't throw exception to prevent activation failure
        }
    }
    
    /**
     * Set Phase 1 default options
     */
    private static function set_phase1_default_options() {
        // Phase 1 comprehensive processing options (enabled by default)
        $phase1_defaults = array(
            'aiox_enable_comprehensive_processing' => true,
            'aiox_auto_extract_ingredients' => true,
            'aiox_auto_update_manifesto' => true,
            'aiox_auto_update_badges' => true,
            'aiox_auto_update_metadata' => true,
            'aiox_phase1_activated' => true,
            'aiox_phase1_version' => '1.0.0',
            'aiox_phase1_activation_date' => current_time('mysql')
        );
        
        foreach ($phase1_defaults as $option => $value) {
            if (get_option($option) === false) {
                add_option($option, $value);
                AIOX_Logger::debug("AIOX Phase 1: Set default option {$option} = " . var_export($value, true));
            }
        }
    }
    
    /**
     * Initialize Phase 1 database options
     */
    private static function initialize_phase1_database() {
        // Initialize sync statistics
        $sync_stats = array(
            'posts_synced' => 0,
            'components_synced' => 0,
            'last_sync' => current_time('mysql'),
            'errors' => array()
        );
        add_option('aiox_sync_stats', $sync_stats);
        
        // Initialize enhanced hook statistics
        $hook_stats = array(
            'posts_processed' => 0,
            'posts_deleted' => 0,
            'comprehensive_processes' => 0,
            'last_activity' => current_time('mysql')
        );
        add_option('aiox_enhanced_hook_stats', $hook_stats);
        
        // Initialize deletion statistics
        $deletion_stats = array(
            'posts_deleted' => 0,
            'components_cleaned' => 0,
            'files_removed' => 0,
            'last_deletion' => null,
            'errors' => array()
        );
        add_option('aiox_deletion_stats', $deletion_stats);
        
        AIOX_Logger::debug('AIOX Phase 1: Initialized database options');
    }
    
    /**
     * Schedule Phase 1 cleanup tasks
     */
    private static function schedule_phase1_tasks() {
        // Schedule weekly cleanup of orphaned data
        if (!wp_next_scheduled('aiox_cleanup_orphaned_data')) {
            wp_schedule_event(time(), 'weekly', 'aiox_cleanup_orphaned_data');
            AIOX_Logger::debug('AIOX Phase 1: Scheduled weekly orphaned data cleanup');
        }
        
        // Schedule daily comprehensive data cleanup
        if (!wp_next_scheduled('aiox_cleanup_comprehensive_data')) {
            wp_schedule_event(time(), 'daily', 'aiox_cleanup_comprehensive_data');
            AIOX_Logger::debug('AIOX Phase 1: Scheduled daily comprehensive data cleanup');
        }
    }
    
    /**
     * Run Phase 1 deactivation tasks
     */
    public static function deactivate() {
        try {
            AIOX_Logger::info('AIOX Phase 1: Starting deactivation tasks');
            
            // Clear Phase 1 scheduled tasks
            wp_clear_scheduled_hook('aiox_cleanup_orphaned_data');
            wp_clear_scheduled_hook('aiox_cleanup_comprehensive_data');
            wp_clear_scheduled_hook('aiox_process_post_comprehensive');
            wp_clear_scheduled_hook('aiox_sync_post_components');
            
            AIOX_Logger::info('AIOX Phase 1: Deactivation tasks completed successfully');
            
        } catch (Exception $e) {
            AIOX_Logger::error('AIOX Phase 1: Deactivation failed', array(
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ));
        }
    }
    
    /**
     * Check if Phase 1 is properly activated
     */
    public static function is_activated() {
        return get_option('aiox_phase1_activated', false);
    }
    
    /**
     * Get Phase 1 activation information
     */
    public static function get_activation_info() {
        return array(
            'activated' => self::is_activated(),
            'version' => get_option('aiox_phase1_version', 'unknown'),
            'activation_date' => get_option('aiox_phase1_activation_date', 'unknown'),
            'comprehensive_processing_enabled' => get_option('aiox_enable_comprehensive_processing', true)
        );
    }
}
