<?php
/**
 * AIOX REST API
 * Provides endpoints for accessing individual post data and licensing information
 */

if (!defined('ABSPATH')) {
    exit;
}

class AIOX_REST_API {
    
    /**
     * Register REST routes
     */
    public static function register_routes() {
        // Posts endpoints
        register_rest_route('aiox/v1', '/posts', array(
            'methods' => 'GET',
            'callback' => array(__CLASS__, 'get_posts_index'),
            'permission_callback' => '__return_true'
        ));
        
        register_rest_route('aiox/v1', '/posts/(?P<id>\d+)', array(
            'methods' => 'GET',
            'callback' => array(__CLASS__, 'get_post'),
            'permission_callback' => '__return_true'
        ));
        
        // License endpoint - returns licensing/monetization configuration
        register_rest_route('aiox/v1', '/license', array(
            'methods' => 'GET',
            'callback' => array(__CLASS__, 'get_license'),
            'permission_callback' => '__return_true'
        ));
        
        // License verification endpoint - check if specific use is allowed
        register_rest_route('aiox/v1', '/license/verify', array(
            'methods' => 'GET',
            'callback' => array(__CLASS__, 'verify_license'),
            'permission_callback' => '__return_true',
            'args' => array(
                'purpose' => array(
                    'required' => false,
                    'type' => 'string',
                    'enum' => array('training', 'citation', 'retrieval', 'analysis', 'commercial'),
                    'description' => 'The intended purpose for content usage'
                ),
                'post_id' => array(
                    'required' => false,
                    'type' => 'integer',
                    'description' => 'Specific post ID to check (optional)'
                )
            )
        ));
        
        // Manifest endpoint - quick access to manifest data
        register_rest_route('aiox/v1', '/manifest', array(
            'methods' => 'GET',
            'callback' => array(__CLASS__, 'get_manifest'),
            'permission_callback' => '__return_true'
        ));
    }
    
    /**
     * Get license information
     * Returns the site's AI licensing and monetization configuration
     */
    public static function get_license($request) {
        $site_name = get_bloginfo('name');
        $site_url = home_url();
        
        // Get preset configurations
        $license_preset = get_option('aiox_license_preset', 'attribution_required');
        $monetization_preset = get_option('aiox_monetization_preset', 'free');
        $enforcement_mode = get_option('aiox_enforcement_mode', 'flexible');
        
        // License presets
        $license_presets = array(
            'open_access' => array(
                'name' => 'Open Access',
                'license_code' => 'AIOX-OA-1.0',
                'training_allowed' => true,
                'citation_required' => true,
                'attribution_required' => true,
                'commercial_use' => true,
                'derivatives_allowed' => true
            ),
            'attribution_required' => array(
                'name' => 'Attribution Required',
                'license_code' => 'AIOX-AR-1.0',
                'training_allowed' => false,
                'citation_required' => true,
                'attribution_required' => true,
                'commercial_use' => true,
                'derivatives_allowed' => true
            ),
            'commercial_license' => array(
                'name' => 'Commercial License',
                'license_code' => 'AIOX-CL-1.0',
                'training_allowed' => false,
                'citation_required' => true,
                'attribution_required' => true,
                'commercial_use' => false,
                'derivatives_allowed' => false
            )
        );
        
        // Monetization presets
        $monetization_presets = array(
            'free' => array(
                'name' => 'Free Access',
                'subscription_required' => false,
                'pay_per_use' => false,
                'rate_limited' => false,
                'pricing_model' => 'free'
            ),
            'freemium' => array(
                'name' => 'Freemium',
                'subscription_required' => false,
                'pay_per_use' => true,
                'rate_limited' => true,
                'pricing_model' => 'freemium'
            ),
            'paid' => array(
                'name' => 'Paid Access',
                'subscription_required' => true,
                'pay_per_use' => true,
                'rate_limited' => true,
                'pricing_model' => 'paid'
            )
        );
        
        // Enforcement modes
        $enforcement_modes = array(
            'strict' => array(
                'name' => 'Strict Enforcement',
                'compliance_level' => 'mandatory',
                'legal_binding' => true,
                'violation_response' => 'block'
            ),
            'flexible' => array(
                'name' => 'Flexible Enforcement',
                'compliance_level' => 'recommended',
                'legal_binding' => false,
                'violation_response' => 'warn'
            )
        );
        
        $license_config = $license_presets[$license_preset] ?? $license_presets['attribution_required'];
        $monetization_config = $monetization_presets[$monetization_preset] ?? $monetization_presets['free'];
        $enforcement_config = $enforcement_modes[$enforcement_mode] ?? $enforcement_modes['flexible'];
        
        // Build response
        $response = array(
            'version' => '1.0',
            'protocol' => 'AIOX',
            'generated_at' => current_time('c'),
            
            'publisher' => array(
                'name' => $site_name,
                'url' => $site_url,
                'contact' => get_option('aiox_license_contact_email', get_option('admin_email')),
                'terms_url' => get_option('aiox_license_terms_url', '')
            ),
            
            'license' => array(
                'preset' => $license_preset,
                'preset_name' => $license_config['name'],
                'license_code' => $license_config['license_code'],
                'permissions' => array(
                    'training' => $license_config['training_allowed'],
                    'citation' => true,
                    'retrieval' => true,
                    'analysis' => true,
                    'commercial' => $license_config['commercial_use'],
                    'derivatives' => $license_config['derivatives_allowed']
                ),
                'requirements' => array(
                    'attribution' => $license_config['attribution_required'],
                    'link_back' => true,
                    'citation_format' => 'Source: ' . $site_name . ' (' . $site_url . ')'
                )
            ),
            
            'monetization' => array(
                'preset' => $monetization_preset,
                'preset_name' => $monetization_config['name'],
                'model' => $monetization_config['pricing_model'],
                'subscription_required' => $monetization_config['subscription_required'],
                'pay_per_use' => $monetization_config['pay_per_use'],
                'rate_limited' => $monetization_config['rate_limited'],
                'pricing' => self::get_pricing_info($monetization_preset)
            ),
            
            'enforcement' => array(
                'mode' => $enforcement_mode,
                'mode_name' => $enforcement_config['name'],
                'compliance_level' => $enforcement_config['compliance_level'],
                'legal_binding' => $enforcement_config['legal_binding'],
                'violation_response' => $enforcement_config['violation_response']
            ),
            
            'custom_terms' => get_option('aiox_license_custom_terms', ''),
            
            'resources' => array(
                'manifest' => $site_url . '/.well-known/aio.json',
                'tdmrep' => $site_url . '/.well-known/tdmrep.json',
                'delta_feed' => $site_url . '/.well-known/aiox.ndjson',
                'terms_page' => get_option('aiox_license_terms_url', ''),
                'api' => array(
                    'license' => rest_url('aiox/v1/license'),
                    'verify' => rest_url('aiox/v1/license/verify'),
                    'posts' => rest_url('aiox/v1/posts'),
                    'manifest' => rest_url('aiox/v1/manifest')
                )
            ),
            
            'stats' => array(
                'total_posts' => self::count_processed_posts(),
                'last_updated' => self::get_last_manifest_update()
            )
        );
        
        return rest_ensure_response($response);
    }
    
    /**
     * Verify if a specific use is allowed
     */
    public static function verify_license($request) {
        $purpose = $request->get_param('purpose');
        $post_id = $request->get_param('post_id');
        
        $license_preset = get_option('aiox_license_preset', 'attribution_required');
        $monetization_preset = get_option('aiox_monetization_preset', 'free');
        $enforcement_mode = get_option('aiox_enforcement_mode', 'flexible');
        
        // Default permissions based on preset
        $permissions = array(
            'open_access' => array(
                'training' => true,
                'citation' => true,
                'retrieval' => true,
                'analysis' => true,
                'commercial' => true
            ),
            'attribution_required' => array(
                'training' => false,
                'citation' => true,
                'retrieval' => true,
                'analysis' => true,
                'commercial' => true
            ),
            'commercial_license' => array(
                'training' => false,
                'citation' => true,
                'retrieval' => true,
                'analysis' => true,
                'commercial' => false
            )
        );
        
        $preset_permissions = $permissions[$license_preset] ?? $permissions['attribution_required'];
        
        // If no specific purpose requested, return all permissions
        if (!$purpose) {
            return rest_ensure_response(array(
                'status' => 'ok',
                'license_preset' => $license_preset,
                'monetization_preset' => $monetization_preset,
                'enforcement_mode' => $enforcement_mode,
                'permissions' => $preset_permissions,
                'requirements' => array(
                    'attribution' => true,
                    'link_back' => true
                ),
                'message' => 'Check individual permissions for specific use cases'
            ));
        }
        
        // Check specific purpose
        $allowed = isset($preset_permissions[$purpose]) ? $preset_permissions[$purpose] : false;
        
        $response = array(
            'purpose' => $purpose,
            'allowed' => $allowed,
            'license_preset' => $license_preset,
            'enforcement_mode' => $enforcement_mode,
            'requirements' => array()
        );
        
        if ($allowed) {
            $response['status'] = 'allowed';
            $response['message'] = sprintf('Usage for "%s" is permitted under current licensing terms', $purpose);
            $response['requirements'] = array(
                'attribution' => true,
                'link_back' => true,
                'citation_format' => 'Source: ' . get_bloginfo('name') . ' (' . home_url() . ')'
            );
        } else {
            $response['status'] = 'denied';
            $response['message'] = sprintf('Usage for "%s" requires licensing agreement', $purpose);
            $response['contact'] = get_option('aiox_license_contact_email', get_option('admin_email'));
            $response['terms_url'] = get_option('aiox_license_terms_url', '');
        }
        
        // Add post-specific info if requested
        if ($post_id) {
            $post = get_post($post_id);
            if ($post) {
                $response['post'] = array(
                    'id' => $post_id,
                    'title' => $post->post_title,
                    'url' => get_permalink($post_id)
                );
            }
        }
        
        return rest_ensure_response($response);
    }
    
    /**
     * Get manifest data via API
     */
    public static function get_manifest($request) {
        $manifest_file = ABSPATH . '.well-known/aio.json';
        
        if (file_exists($manifest_file)) {
            $content = file_get_contents($manifest_file);
            $data = json_decode($content, true);
            
            if ($data) {
                return rest_ensure_response($data);
            }
        }
        
        // Return basic info if manifest doesn't exist
        return rest_ensure_response(array(
            'error' => false,
            'message' => 'Manifest file not yet generated',
            'publisher' => array(
                'name' => get_bloginfo('name'),
                'url' => home_url()
            ),
            'license_api' => rest_url('aiox/v1/license')
        ));
    }
    
    /**
     * Get pricing info based on monetization preset
     */
    private static function get_pricing_info($preset) {
        $pricing = array(
            'free' => array(
                'type' => 'free',
                'description' => 'No cost for AI access',
                'tiers' => array(
                    array('name' => 'Free', 'price' => 0, 'requests' => 'unlimited')
                )
            ),
            'freemium' => array(
                'type' => 'freemium',
                'description' => 'Free tier available, premium features require payment',
                'tiers' => array(
                    array('name' => 'Free', 'price' => 0, 'requests' => '1000/month'),
                    array('name' => 'Pro', 'price' => 'contact', 'requests' => 'unlimited')
                ),
                'contact_for_pricing' => true
            ),
            'paid' => array(
                'type' => 'paid',
                'description' => 'All access requires licensing agreement',
                'tiers' => array(
                    array('name' => 'Enterprise', 'price' => 'contact', 'requests' => 'negotiable')
                ),
                'contact_for_pricing' => true
            )
        );
        
        return $pricing[$preset] ?? $pricing['free'];
    }
    
    /**
     * Count processed posts
     */
    private static function count_processed_posts() {
        global $wpdb;
        return (int) $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->postmeta} WHERE meta_key = '_aiox_processed' AND meta_value = '1'");
    }
    
    /**
     * Get last manifest update time
     */
    private static function get_last_manifest_update() {
        $manifest_file = ABSPATH . '.well-known/aio.json';
        if (file_exists($manifest_file)) {
            return date('c', filemtime($manifest_file));
        }
        return null;
    }
    
    /**
     * Get posts index
     */
    public static function get_posts_index($request) {
        $per_page = $request->get_param('per_page') ?? 20;
        $page = $request->get_param('page') ?? 1;
        
        $posts = get_posts(array(
            'post_type' => array('post', 'page'),
            'post_status' => 'publish',
            'posts_per_page' => $per_page,
            'paged' => $page,
            'meta_query' => array(
                array(
                    'key' => '_aiox_processed',
                    'value' => '1',
                    'compare' => '='
                )
            )
        ));
        
        $items = array();
        $home_url = home_url();
        
        foreach ($posts as $post) {
            $items[] = array(
                'id' => $post->ID,
                'title' => $post->post_title,
                'url' => get_permalink($post->ID),
                'detail_url' => $home_url . '/.well-known/posts/' . $post->ID . '.json',
                'api_url' => rest_url('aiox/v1/posts/' . $post->ID)
            );
        }
        
        return rest_ensure_response($items);
    }
    
    /**
     * Get individual post
     */
    public static function get_post($request) {
        $post_id = $request->get_param('id');
        
        // Try to read from file first
        $well_known_dir = ABSPATH . '.well-known';
        $post_file = $well_known_dir . '/posts/' . $post_id . '.json';
        
        if (file_exists($post_file)) {
            $content = file_get_contents($post_file);
            $data = json_decode($content, true);
            
            if ($data) {
                return rest_ensure_response($data);
            }
        }
        
        // Fallback: generate on the fly
        $post = get_post($post_id);
        
        if (!$post) {
            return new WP_Error('not_found', 'Post not found', array('status' => 404));
        }
        
        // Check if processed
        $processed = get_post_meta($post_id, '_aiox_processed', true);
        
        if ($processed !== '1') {
            return new WP_Error('not_processed', 'Post not processed by AIOX', array('status' => 404));
        }
        
        // Return basic data if file doesn't exist
        return rest_ensure_response(array(
            'post_id' => $post_id,
            'title' => $post->post_title,
            'url' => get_permalink($post_id),
            'note' => 'Complete data file not yet generated. Please run backfill tool.'
        ));
    }
}

// Register REST routes
add_action('rest_api_init', array('AIOX_REST_API', 'register_routes'));