<?php
/**
 * AIOX Badge Class - Redesigned
 * 
 * Handles the AI-friendly badge display and interaction
 * Version 2.1.0 - Clean, modern aesthetic redesign
 */

if (!defined('ABSPATH')) {
    exit;
}

class AIOX_Badge {
    
    /**
     * Decode HTML entities properly to avoid double-encoding
     * Handles cases where content was stored with encoded entities
     * 
     * @param string $text Text to decode
     * @param bool $escape Whether to escape for HTML output (default: true)
     * @return string Properly decoded text
     */
    private static function decode_text($text, $escape = true) {
        if (empty($text)) {
            return '';
        }
        
        // Decode HTML entities multiple times to handle double/triple encoding
        $decoded = $text;
        $previous = '';
        $max_iterations = 5; // Prevent infinite loop
        $i = 0;
        
        while ($decoded !== $previous && $i < $max_iterations) {
            $previous = $decoded;
            $decoded = html_entity_decode($decoded, ENT_QUOTES | ENT_HTML5, 'UTF-8');
            $i++;
        }
        
        // Optionally escape for HTML output
        return $escape ? esc_html($decoded) : $decoded;
    }
    
    /**
     * Constructor
     */
    public function __construct() {
        add_action('wp_footer', array($this, 'render_badge'));
        add_action('wp_enqueue_scripts', array($this, 'enqueue_scripts'));
        add_action('wp_ajax_aiox_get_qa', array($this, 'ajax_get_qa'));
        add_action('wp_ajax_nopriv_aiox_get_qa', array($this, 'ajax_get_qa'));
    }
    
    /**
     * Enqueue frontend scripts and styles
     */
    public function enqueue_scripts() {
        if (!is_singular() || !AIOX_Utils::is_feature_enabled('ai_badge')) {
            return;
        }
        
        wp_enqueue_style(
            'aiox-badge',
            AIOX_PLUGIN_URL . 'assets/css/badge.css',
            array(),
            AIOX_VERSION
        );
        
        wp_enqueue_script(
            'aiox-badge',
            AIOX_PLUGIN_URL . 'assets/js/badge.js',
            array('jquery'),
            AIOX_VERSION,
            true
        );
        
        wp_localize_script('aiox-badge', 'aiox_badge', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('aiox_badge_nonce'),
            'post_id' => get_the_ID(),
            'strings' => array(
                'loading' => __('Loading...', 'aiox-publisher-suite-pro'),
                'error' => __('Failed to load content', 'aiox-publisher-suite-pro'),
                'no_content' => __('No AI-friendly content available', 'aiox-publisher-suite-pro'),
                'close' => __('Close', 'aiox-publisher-suite-pro'),
                'ai_content' => __('AI-Friendly Content', 'aiox-publisher-suite-pro')
            )
        ));
    }
    
    /**
     * Render the AI badge with custom settings
     */
    public function render_badge() {
        if (!is_singular() || !AIOX_Utils::is_feature_enabled('ai_badge')) {
            return;
        }
        
        $post_id = get_the_ID();
        
        // Check if badge is enabled
        if (!get_option('aiox_badge_enabled', true)) {
            return;
        }
        
        // Version 2.0.1+: Only show badge if content is already AI-processed
        $processed_data = get_post_meta($post_id, '_aiox_processed_data', true);
        $is_ai_enhanced = $processed_data && isset($processed_data['processing_method']) && $processed_data['processing_method'] !== 'basic';
        
        // Don't show badge if content is not AI-processed
        if (!$is_ai_enhanced) {
            return;
        }
        
        // Check if we have Q&A content to display
        if (!isset($processed_data['qa_pairs']) || empty($processed_data['qa_pairs'])) {
            return;
        }
        
        // Get custom settings - all defined by the system
        $position = get_option('aiox_badge_position', 'bottom-right');
        $shape = get_option('aiox_badge_shape', 'rounded');
        $size = get_option('aiox_badge_size', 'medium');
        $bg_color = get_option('aiox_badge_bg_color', '#667eea');
        $text_color = get_option('aiox_badge_text_color', '#ffffff');
        $title = get_option('aiox_badge_title', 'AI-Enhanced');
        $subtitle = get_option('aiox_badge_subtitle', 'AI Powered');
        
        // Calculate border radius based on shape setting
        $border_radius = '10px';
        if ($shape === 'square') $border_radius = '0';
        if ($shape === 'pill') $border_radius = '50px';
        if ($shape === 'circle') $border_radius = '50%';
        if ($shape === 'rounded') $border_radius = '10px';
        
        // Size configurations - more compact
        $padding = '10px 14px';
        $title_size = '14px';
        $subtitle_size = '11px';
        
        if ($size === 'small') {
            $padding = '8px 12px';
            $title_size = '12px';
            $subtitle_size = '10px';
        } elseif ($size === 'large') {
            $padding = '14px 18px';
            $title_size = '16px';
            $subtitle_size = '12px';
        }
        
        // Position styles
        $position_styles = [
            'bottom-right' => 'bottom: 24px; right: 24px;',
            'bottom-left' => 'bottom: 24px; left: 24px;',
            'top-right' => 'top: 24px; right: 24px;',
            'top-left' => 'top: 24px; left: 24px;'
        ];
        $position_style = $position_styles[$position] ?? $position_styles['bottom-right'];
        
        $method = $processed_data['processing_method'] ?? 'basic';
        
        // Get ingredients count - check multiple sources
        $ingredients_count = 0;
        if (isset($processed_data['ingredients']) && is_array($processed_data['ingredients'])) {
            $ingredients_count = count($processed_data['ingredients']);
        }
        // If not in processed_data, check separate _aiox_ingredients meta
        if ($ingredients_count === 0) {
            $ingredients_data = get_post_meta($post_id, '_aiox_ingredients', true);
            if (!empty($ingredients_data)) {
                $ingredients = is_string($ingredients_data) ? json_decode($ingredients_data, true) : $ingredients_data;
                if (is_array($ingredients)) {
                    $ingredients_count = count($ingredients);
                }
            }
        }
        // Also check _aiox_ingredients_count meta if available
        if ($ingredients_count === 0) {
            $stored_count = get_post_meta($post_id, '_aiox_ingredients_count', true);
            if (!empty($stored_count) && is_numeric($stored_count)) {
                $ingredients_count = intval($stored_count);
            }
        }
        
        // Prepare Q&A data
        $qa_pairs = $processed_data['qa_pairs'];
        $post = get_post($post_id);
        
        // Build AIOX native data structure
        $aiox_data = [
            '@context' => 'https://aiox.io/context/v1',
            '@type' => 'AIOX.Badge',
            '@id' => get_permalink($post_id) . '#aiox-badge',
            'version' => AIOX_VERSION,
            'processingMethod' => $method,
            'processedAt' => get_post_meta($post_id, '_aiox_processed_at', true),
            'ingredientsCount' => $ingredients_count,
            'content' => [
                '@type' => 'AIOX.ContentMetadata',
                'url' => get_permalink($post_id),
                'title' => get_the_title($post_id),
                'summary' => $processed_data['summary'] ?? '',
                'intent' => $processed_data['metadata']['intent'] ?? null,
                'tone' => $processed_data['metadata']['tone'] ?? null,
                'orientation' => $processed_data['metadata']['orientation'] ?? null,
                'domain' => $processed_data['metadata']['domain'] ?? null,
                'utility' => $processed_data['metadata']['utility'] ?? null,
                'targetAudience' => $processed_data['metadata']['target_audience'] ?? null,
                'narrativeType' => $processed_data['metadata']['narrative_type'] ?? null,
                'mediaType' => $processed_data['metadata']['media_type'] ?? null,
                'geolocationSensitivity' => $processed_data['metadata']['geolocation_sensitivity'] ?? null,
                'marketSegments' => $processed_data['metadata']['market_segments'] ?? null
            ],
            'qaData' => [
                '@type' => 'AIOX.QuestionAnswerSet',
                'totalPairs' => count($qa_pairs),
                'pairs' => array_map(function($qa, $index) {
                    return [
                        '@type' => 'AIOX.QAPair',
                        'id' => 'qa-' . ($index + 1),
                        'question' => html_entity_decode($qa['question'] ?? '', ENT_QUOTES, 'UTF-8'),
                        'answer' => html_entity_decode($qa['answer'] ?? '', ENT_QUOTES, 'UTF-8'),
                        'confidence' => $qa['confidence'] ?? null,
                        'source' => $qa['source'] ?? 'ai_extraction'
                    ];
                }, $qa_pairs, array_keys($qa_pairs))
            ],
            'manifesto' => [
                '@type' => 'AIOX.ManifestoReference',
                'url' => home_url('/.well-known/aio.json'),
                'version' => '1.0',
                'description' => 'Publisher policies and content structure'
            ]
        ];
        
        // Remove null values from content metadata
        $aiox_data['content'] = array_filter($aiox_data['content'], function($v) { 
            return $v !== null; 
        });
        
        // Build JSON-LD structured data (Schema.org FAQPage - Compatibility Layer)
        $faq_schema = [
            '@context' => 'https://schema.org',
            '@type' => 'FAQPage',
            'mainEntity' => []
        ];
        
        foreach ($qa_pairs as $qa) {
            if (is_array($qa) && isset($qa['question']) && isset($qa['answer'])) {
                $faq_schema['mainEntity'][] = [
                    '@type' => 'Question',
                    'name' => html_entity_decode($qa['question'], ENT_QUOTES, 'UTF-8'),
                    'acceptedAnswer' => [
                        '@type' => 'Answer',
                        'text' => html_entity_decode($qa['answer'], ENT_QUOTES, 'UTF-8')
                    ]
                ];
            }
        }
        
        // Output AIOX Native Format (Primary for AI systems)
        ?>
        <!-- AIOX Badge Data: Native Format -->
        <!-- For AI Systems: https://aiox.io/standard/v1 -->
        <script type="application/x-aiox+json">
        <?php echo wp_json_encode($aiox_data, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE); ?>
        </script>
        
        <!-- Schema.org Compatibility Layer -->
        <script type="application/ld+json">
        <?php echo wp_json_encode($faq_schema, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES); ?>
        </script>
        
        <!-- Machine-Readable Q&A Content for AI Crawlers -->
        <!-- This content is visually hidden but accessible to crawlers and screen readers -->
        <!-- Using clip-rect method (SEO-safe) instead of position:absolute/left:-9999px (can be penalized) -->
        <div id="aiox-machine-readable-qa-<?php echo esc_attr($post_id); ?>" 
             class="aiox-machine-readable-content" 
             style="clip: rect(0 0 0 0); clip-path: inset(50%); height: 1px; width: 1px; margin: -1px; overflow: hidden; padding: 0; position: absolute; white-space: nowrap;"
             itemscope itemtype="https://schema.org/FAQPage"
             role="region"
             aria-label="AI-Enhanced Content Summary">
            
            <meta itemprop="name" content="<?php echo esc_attr(get_the_title($post_id)); ?>" />
            <meta itemprop="url" content="<?php echo esc_url(get_permalink($post_id)); ?>" />
            
            <div class="aiox-qa-metadata">
                <p><strong>Content Type:</strong> AI-Enhanced Article</p>
                <p><strong>Processing Method:</strong> <?php echo self::decode_text($method); ?></p>
                <p><strong>Processed At:</strong> <?php echo esc_html(get_post_meta($post_id, '_aiox_processed_at', true)); ?></p>
                <?php if (isset($processed_data['summary'])): ?>
                <p><strong>Summary:</strong> <?php echo self::decode_text($processed_data['summary']); ?></p>
                <?php endif; ?>
            </div>
            
            <?php foreach ($qa_pairs as $index => $qa): 
                if (!is_array($qa) || !isset($qa['question']) || !isset($qa['answer'])) continue;
                $question_text = self::decode_text($qa['question']);
                $answer_text = self::decode_text($qa['answer']);
            ?>
            <div class="aiox-qa-pair" 
                 itemprop="mainEntity" 
                 itemscope 
                 itemtype="https://schema.org/Question">
                
                <h3 itemprop="name" class="aiox-question">
                    Q<?php echo ($index + 1); ?>: <?php echo $question_text; ?>
                </h3>
                
                <div itemprop="acceptedAnswer" 
                     itemscope 
                     itemtype="https://schema.org/Answer">
                    <div itemprop="text" class="aiox-answer">
                        <?php echo $answer_text; ?>
                    </div>
                    <?php if (isset($qa['confidence'])): ?>
                    <meta itemprop="upvoteCount" content="<?php echo esc_attr(round($qa['confidence'] * 100)); ?>" />
                    <?php endif; ?>
                </div>
            </div>
            <?php endforeach; ?>
        </div>
        
        <!-- AIOX AI Badge with Custom Styling -->
        <!-- For AI Crawlers: See JSON-LD and #aiox-machine-readable-qa-<?php echo esc_attr($post_id); ?> above for structured Q&A data -->
        <div id="aiox-badge-<?php echo esc_attr($post_id); ?>" 
             class="aiox-badge aiox-badge-custom" 
             data-post-id="<?php echo esc_attr($post_id); ?>"
             data-nosnippet
             style="position: fixed; z-index: 9999; <?php echo $position_style; ?>"
             aria-label="<?php esc_attr_e('AI Content Information', 'aiox-publisher-suite-pro'); ?>">
            
            <!-- Trigger Button - Compact, text only, width fits content -->
            <div class="aiox-badge-trigger" 
                 style="background: <?php echo esc_attr($bg_color); ?>; 
                        color: <?php echo esc_attr($text_color); ?>; 
                        border-radius: <?php echo esc_attr($border_radius); ?>; 
                        padding: 10px 20px; 
                        cursor: pointer; 
                        box-shadow: 0 4px 12px rgba(0,0,0,0.15);
                        transition: all 0.3s ease;
                        text-align: center;
                        width: auto;
                        max-width: fit-content;
                        font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;">
                <div style="font-size: <?php echo esc_attr($title_size); ?>; font-weight: 600; line-height: 1.2;"><?php echo self::decode_text($title); ?></div>
                <?php if (!empty($subtitle)) : ?><div style="opacity: 0.85; font-size: <?php echo esc_attr($subtitle_size); ?>; font-weight: 400; margin-top: 2px; line-height: 1.2;"><?php echo self::decode_text($subtitle); ?></div><?php endif; ?>
            </div>
            
            <!-- Content Panel - Original Layout -->
            <div class="aiox-badge-content" style="display: none; position: absolute; background: white; border-radius: 12px; box-shadow: 0 10px 40px rgba(0,0,0,0.15); width: 380px; max-width: calc(100vw - 40px); <?php echo strpos($position, 'right') !== false ? 'right: 0;' : 'left: 0;'; ?> <?php echo strpos($position, 'top') !== false ? 'top: calc(100% + 12px);' : 'bottom: calc(100% + 12px);'; ?> font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif; overflow: hidden;">
                
                <!-- Header with gradient -->
                <div class="aiox-badge-header" style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); padding: 14px 18px; position: relative;">
                    <h3 style="margin: 0; font-size: 16px; font-weight: 600; color: #ffffff; letter-spacing: -0.01em;"><?php _e('AI-FRIENDLY CONTENT', 'aiox-publisher-suite-pro'); ?></h3>
                    <button class="aiox-badge-close" type="button" style="position: absolute; top: 10px; right: 12px; background: rgba(255,255,255,0.2); border: none; border-radius: 6px; width: 28px; height: 28px; cursor: pointer; display: flex; align-items: center; justify-content: center; transition: background 0.2s ease;">
                        <svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="#ffffff" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <line x1="18" y1="6" x2="6" y2="18"/>
                            <line x1="6" y1="6" x2="18" y2="18"/>
                        </svg>
                    </button>
                </div>
                
                <div class="aiox-badge-body" style="padding: 0; max-height: 380px; overflow-y: auto;">
                    
                    <!-- Loading State -->
                    <div class="aiox-badge-loading" style="display: none; flex-direction: column; align-items: center; justify-content: center; padding: 24px 16px;">
                        <div class="aiox-spinner" style="width: 28px; height: 28px; border: 3px solid #e5e7eb; border-top-color: #14b8a6; border-radius: 50%; animation: aiox-spin 0.8s linear infinite;"></div>
                        <p style="margin-top: 10px; color: #6b7280; font-size: 13px;"><?php _e('Loading AI-friendly content...', 'aiox-publisher-suite-pro'); ?></p>
                    </div>
                    
                    <!-- Q&A List - Content populated by badge.js -->
                    <div class="aiox-badge-qa-list" style="display: none; padding: 14px 16px;">
                        <!-- Content will be loaded here by badge.js -->
                    </div>
                    
                    <!-- Error State -->
                    <div class="aiox-badge-error" style="display: none; text-align: center; padding: 24px 16px;">
                        <p style="color: #6b7280; font-size: 13px; margin-bottom: 12px;"><?php _e('Failed to load content. Please try again.', 'aiox-publisher-suite-pro'); ?></p>
                        <button class="aiox-retry-btn" type="button" style="background: #14b8a6; color: #ffffff; border: none; padding: 8px 16px; border-radius: 6px; font-size: 13px; font-weight: 500; cursor: pointer;"><?php _e('Retry', 'aiox-publisher-suite-pro'); ?></button>
                    </div>
                </div>
                
                <!-- Footer -->
                <div class="aiox-badge-footer" style="padding: 10px 16px; background: #f9fafb; border-top: 1px solid #e5e7eb; display: flex; justify-content: space-between; align-items: center;">
                    <span style="display: inline-flex; align-items: center; gap: 5px; background: #14b8a6; color: #ffffff; padding: 4px 10px; border-radius: 14px; font-size: 11px; font-weight: 500;">
                        <svg width="10" height="10" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="3" stroke-linecap="round" stroke-linejoin="round">
                            <polyline points="20 6 9 17 4 12"/>
                        </svg>
                        <?php _e('AI-Optimized', 'aiox-publisher-suite-pro'); ?>
                    </span>
                    <a href="https://aioxsuite.com" target="_blank" rel="noopener noreferrer" style="color: #6b7280; text-decoration: none; font-size: 11px; transition: color 0.2s ease;">
                        <?php _e('Powered by AIOX', 'aiox-publisher-suite-pro'); ?>
                    </a>
                </div>
            </div>
        </div>
        
        <style>
        @keyframes aiox-spin {
            to { transform: rotate(360deg); }
        }
        .aiox-badge-trigger:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(0, 0, 0, 0.3) !important;
        }
        .aiox-badge-close:hover {
            background: rgba(255,255,255,0.3) !important;
        }
        .aiox-badge-footer a:hover {
            color: #14b8a6 !important;
        }
        .aiox-badge-body::-webkit-scrollbar {
            width: 6px;
        }
        .aiox-badge-body::-webkit-scrollbar-track {
            background: #f1f5f9;
        }
        .aiox-badge-body::-webkit-scrollbar-thumb {
            background: #cbd5e1;
            border-radius: 3px;
        }
        .aiox-badge-body::-webkit-scrollbar-thumb:hover {
            background: #94a3b8;
        }
        </style>
        <?php
    }
    
    /**
     * AJAX handler to get Q&A content
     */
    public function ajax_get_qa() {
        check_ajax_referer('aiox_badge_nonce', 'nonce');
        
        $post_id = intval($_POST['post_id'] ?? 0);
        
        if (!$post_id) {
            wp_send_json_error('Invalid post ID');
            return;
        }
        
        // First check for AI-enhanced processed data
        $processed_data = get_post_meta($post_id, '_aiox_processed_data', true);
        $is_ai_enhanced = $processed_data && isset($processed_data['processing_method']) && $processed_data['processing_method'] !== 'basic';
        
        $qa_pairs = array();
        $method = 'basic';
        
        // Version 2.0.1+: Badge only appears for already AI-processed content
        if ($is_ai_enhanced && isset($processed_data['qa_pairs'])) {
            // Use AI-enhanced Q&A pairs
            $qa_pairs = $processed_data['qa_pairs'];
            $method = $processed_data['processing_method'];
        } else {
            // This should not happen since badge only renders for AI-processed content
            wp_send_json_error('Content is not AI-processed. Badge should not be visible.');
            return;
        }
        
        // Format Q&A for display (no HTML escaping for JSON output)
        $formatted_qa = array();
        foreach ($qa_pairs as $qa) {
            if (is_array($qa)) {
                $formatted_qa[] = array(
                    'id' => isset($qa['id']) ? $qa['id'] : uniqid(),
                    'question' => self::decode_text($qa['question'], false),
                    'answer' => self::decode_text($qa['answer'], false),
                    'confidence' => isset($qa['confidence']) ? $qa['confidence'] : 0.8,
                    'source' => isset($qa['source']) ? $qa['source'] : ($is_ai_enhanced ? 'AI Enhanced' : 'Basic Extraction')
                );
            }
        }
        
        // Get post metadata
        $post = get_post($post_id);
        $metadata = array(
            'title' => get_the_title($post_id),
            'url' => get_permalink($post_id),
            'processed_at' => get_post_meta($post_id, '_aiox_processed_at', true),
            'method' => $method,
            'is_ai_enhanced' => $is_ai_enhanced,
            'qa_count' => count($formatted_qa),
            'processing_details' => $is_ai_enhanced ? array(
                'summary' => isset($processed_data['summary']) ? self::decode_text($processed_data['summary'], false) : '',
                'topics' => isset($processed_data['topics']) ? array_slice($processed_data['topics'], 0, 5) : array(),
                'sentiment' => isset($processed_data['sentiment']) ? $processed_data['sentiment'] : '',
                'content_type' => isset($processed_data['content_type']) ? $processed_data['content_type'] : ''
            ) : null
        );
        
        wp_send_json_success(array(
            'qa_pairs' => $formatted_qa,
            'metadata' => $metadata
        ));
    }
    
    /**
     * Get badge position from settings
     */
    private function get_badge_position() {
        $position = AIOX_Utils::get_option('badge_position', 'bottom-right');
        
        $positions = array(
            'bottom-right' => array('bottom' => '24px', 'right' => '24px'),
            'bottom-left' => array('bottom' => '24px', 'left' => '24px'),
            'top-right' => array('top' => '24px', 'right' => '24px'),
            'top-left' => array('top' => '24px', 'left' => '24px')
        );
        
        return $positions[$position] ?? $positions['bottom-right'];
    }
    
    /**
     * Check if badge should be shown on current page
     */
    private function should_show_badge() {
        // Don't show on admin pages
        if (is_admin()) {
            return false;
        }
        
        // Only show on singular posts/pages
        if (!is_singular()) {
            return false;
        }
        
        // Check if feature is enabled
        if (!AIOX_Utils::is_feature_enabled('ai_badge')) {
            return false;
        }
        
        // Check if post type is supported
        $post_type = get_post_type();
        if (!AIOX_Utils::should_process_post_type($post_type)) {
            return false;
        }
        
        // Version 2.0.1+: Only show badge for AI-processed content
        $post_id = get_the_ID();
        $processed_data = get_post_meta($post_id, '_aiox_processed_data', true);
        $is_ai_enhanced = $processed_data && isset($processed_data['processing_method']) && $processed_data['processing_method'] !== 'basic';
        
        return $is_ai_enhanced;
    }
}